package com.arms.egovframework.javaservice.esframework.repository.common;

import com.arms.api.issue.almapi.model.entity.AlmIssueEntity;
import com.arms.egovframework.javaservice.esframework.annotation.*;
import com.arms.egovframework.javaservice.esframework.esquery.SimpleQuery;
import com.arms.egovframework.javaservice.esframework.factory.SearchDocQueryFactory;
import com.arms.egovframework.javaservice.esframework.model.dto.esquery.SearchDocDTO;
import com.arms.egovframework.javaservice.esframework.model.vo.CatIndexVO;
import com.arms.egovframework.javaservice.esframework.model.vo.DocumentAggregations;
import com.arms.egovframework.javaservice.esframework.model.vo.DocumentResultWrapper;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.extern.slf4j.Slf4j;
import org.apache.http.util.EntityUtils;
import org.opensearch.client.Request;
import org.opensearch.client.RequestOptions;
import org.opensearch.client.Response;
import org.opensearch.client.RestHighLevelClient;
import org.opensearch.common.unit.TimeValue;
import org.opensearch.data.client.orhlc.NativeSearchQuery;
import org.opensearch.data.client.orhlc.NativeSearchQueryBuilder;
import org.opensearch.data.client.orhlc.OpenSearchRestTemplate;
import org.opensearch.data.core.OpenSearchOperations;
import org.opensearch.index.query.BoolQueryBuilder;
import org.opensearch.index.query.QueryBuilders;
import org.opensearch.index.query.TermsQueryBuilder;
import org.opensearch.index.reindex.ReindexRequest;
import org.springframework.core.annotation.AnnotationUtils;
import org.springframework.data.annotation.Id;
import org.springframework.data.elasticsearch.NoSuchIndexException;
import org.springframework.data.elasticsearch.annotations.Document;
import org.springframework.data.elasticsearch.annotations.Field;
import org.springframework.data.elasticsearch.core.*;
import org.springframework.data.elasticsearch.core.mapping.IndexCoordinates;
import org.springframework.data.elasticsearch.core.query.Query;
import org.springframework.data.elasticsearch.repository.support.ElasticsearchEntityInformation;

import java.io.Serializable;
import java.lang.reflect.Constructor;
import java.lang.reflect.Method;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import java.util.stream.StreamSupport;

import static com.arms.api.util.ApplicationContextProvider.getBean;
import static com.arms.egovframework.javaservice.esframework.util.ReflectionUtil.*;
import static java.util.stream.Collectors.*;

/*
	ApplicationContextAware 를 사용할때에는 순환참조를 사용 하는게 아닌지 반드시 확인해야한다.
*/
@Slf4j
public class EsCommonRepositoryImpl<T, U extends Serializable> extends SimpleElasticsearchWrapperRepository<T, U> {

    private final Class<T> entityClazz;

    private final OpenSearchOperations openSearchOperations;

    private final SimpleDateFormat defaultFormater;

    public EsCommonRepositoryImpl(ElasticsearchEntityInformation<T, U> metadata
            , OpenSearchOperations openSearchOperations) {

        super(metadata, openSearchOperations);

        this.entityClazz = metadata.getJavaType();

        this.openSearchOperations = openSearchOperations;

        this.defaultFormater = new SimpleDateFormat(this.findDateFormat(rollingIndexNameSuffix()));

    }

    @Override
    public DocumentAggregations aggregateRecentDocs(Query query) {
        return new DocumentAggregations(openSearchOperations.search(query, entityClazz));
    }

    @Override
    public DocumentAggregations aggregateDocs(Query query) {
        return new DocumentAggregations(openSearchOperations.search(query, entityClazz));
    }

    @Override
    public DocumentResultWrapper<T> findAllHits() {
        NativeSearchQueryBuilder nativeSearchQueryBuilder = new NativeSearchQueryBuilder();
        nativeSearchQueryBuilder.withMaxResults(10000);
        return new DocumentResultWrapper<>(openSearchOperations.search(nativeSearchQueryBuilder.build(), entityClazz),
            entityClazz);
    }

    @Override
    public DocumentResultWrapper<T> findHits(Query query) {
        return new DocumentResultWrapper<>(openSearchOperations.search(query, entityClazz), entityClazz);
    }

    @Override
    public DocumentResultWrapper<T> findRecentAllHits() {

        String recentFieldName = Optional.of(fieldInfo(entityClazz, Recent.class).getAnnotation(Field.class).name())
            .filter(s->!s.isEmpty())
            .orElse(fieldInfo(entityClazz, Recent.class).getName());

        TermsQueryBuilder termsQueryBuilder = QueryBuilders.termsQuery(recentFieldName, true);

        BoolQueryBuilder boolQueryBuilder = QueryBuilders.boolQuery();
        boolQueryBuilder.must(termsQueryBuilder);

        NativeSearchQueryBuilder nativeSearchQueryBuilder = new NativeSearchQueryBuilder();
        nativeSearchQueryBuilder.withQuery(boolQueryBuilder);
        nativeSearchQueryBuilder.withMaxResults(10000);

        return new DocumentResultWrapper<>(openSearchOperations.search(nativeSearchQueryBuilder.build(), entityClazz),
            entityClazz);
    }

    @Override
    public DocumentResultWrapper<T> findRecentHits(Query query) {

        try{
            return new DocumentResultWrapper<>(
                openSearchOperations.search(query, entityClazz), entityClazz);
        } catch (NoSuchIndexException e) {
            String errorMessage = e.getMessage();
            if (errorMessage != null && errorMessage.contains("no such index")) {
                throw new IllegalArgumentException(e.getMessage());
            }
            throw e;
        } catch (Exception e){
            throw new IllegalArgumentException(e.getMessage());
        }
    }

    @Override
    public T findRecentDocByRecentId(U u) {

        try{

            String recentIdFieldName = Optional.of(fieldInfo(entityClazz, RecentId.class).getAnnotation(Field.class).name())
                .filter(s->!s.isEmpty())
                .orElse(fieldInfo(entityClazz, RecentId.class).getName());

            String recentFieldName = Optional.of(fieldInfo(entityClazz, Recent.class).getAnnotation(Field.class).name())
                .filter(s->!s.isEmpty())
                .orElse(fieldInfo(entityClazz, Recent.class).getName());

            SearchDocDTO searchDocDTO = SimpleQuery.termQueryFilter(recentIdFieldName, u).andTermQueryFilter(recentFieldName, true).toSearchDoc();

            DocumentResultWrapper<T> documentSearchResult
                    = new DocumentResultWrapper<>(openSearchOperations.search(
                SearchDocQueryFactory.searchDoc(searchDocDTO).create(), entityClazz), entityClazz);
            return documentSearchResult.fetchOnlyOne();

        } catch (NoSuchIndexException e) {
            String errorMessage = e.getMessage();
            if (errorMessage != null && errorMessage.contains("no such index")) {
                throw new IllegalArgumentException(e.getMessage());
            }
            throw e;
        } catch (Exception e){
            throw new IllegalArgumentException(e.getMessage());
        }
    }

    @Override
    public T findDocById(U u) {
        return openSearchOperations.get(String.valueOf(u), entityClazz);
    }

    @Override
    public DocumentResultWrapper<T> findRecentDocById(U u) {
        Query query = openSearchOperations.idsQuery(List.of(String.valueOf(u)));
        return new DocumentResultWrapper<>(openSearchOperations.search(query, entityClazz),entityClazz);
    }

    @Override
    public <S extends T> S saveEmpty(S entity){
        return openSearchOperations.save(entity, rollingIndexName());
    }

    @Override
    public <S extends T> Iterable<S> saveAll(Iterable<S> entities){
        return StreamSupport.stream(entities.spliterator(), false)
                .toList().stream()
                .map(this::save).collect(toList());
    }

    @Override
    public <S extends T> S modifyWithIndexName(S entity, String indexName){
        return openSearchOperations.save(entity, IndexCoordinates.of(indexName));
    }

    @Override
    public <S extends T> S  save(S entity){

        ElasticSearchIndex elasticSearchIndexAnnotation = AnnotationUtils.findAnnotation(entityClazz, ElasticSearchIndex.class);

        if(isExistFieldWithAnnotation(entityClass, ElasticSearchCreatedDate.class)){

            String createDateName = fieldInfo(entityClazz, ElasticSearchCreatedDate.class).getName();

            Date value = (Date) getValue(entity, createDateName);

            if(value==null){
                setValue(entity, createDateName, generateDate(new Date()));
            }

        }

        if(isExistFieldWithAnnotation(entityClass, ElasticSearchUpdateDate.class)){

            String updateDateName = fieldInfo(entityClazz, ElasticSearchUpdateDate.class).getName();
            setValue(entity, updateDateName, generateDate(new Date()));

        }

        if(elasticSearchIndexAnnotation==null){
            return openSearchOperations.save(entity);
        }

        if(isExistFieldWithAnnotation(entityClass, Id.class)){
            String idName = fieldInfo(entityClazz, Id.class).getName();
            setValue(entity, idName, null);//아이디 초기화 하여 중복된 아이디 발생 방지
        }

        Iterable<S> s = this.saveSupport(List.of(entity));

        if(!s.iterator().hasNext()){
            log.info("저장할 데이터가 없습니다.");
        }

        return entity;
    }

    private Date generateDate(Date date) {
        // 타임스탬프를 UTC+9로 변환
        LocalDateTime localDateTime = LocalDateTime.ofInstant(date.toInstant(), ZoneId.of("UTC+9"));
        ZonedDateTime zonedDateTime = ZonedDateTime.of(localDateTime, ZoneId.of("UTC+9"));
        return Date.from(zonedDateTime.toInstant());
    }

    private  <S extends T> Iterable<S> saveSupport(Iterable<S> entities) {

        log.info("[ 공통저장소_구현체 :: saveSupport ] entities = {}",entities);

        RecentFieldConvertor<S,T> recentFieldConvertor;

        try {
            recentFieldConvertor = new RecentFieldConvertor<>(entities,entityClass,openSearchOperations);
        } catch (Exception e) {
            log.info(e.getMessage());
            return entities;
        }

        Map<String, List<T>> recentFalseListIfNotEqual
            = recentFieldConvertor.listMapGroupByIndex(recentFieldConvertor::setRecentFalseIfNotEqual);

        saveRecentFalseList(recentFalseListIfNotEqual);

        Map<String, List<T>> recentFalseListIfDuplicate
            = recentFieldConvertor.listMapGroupByIndex(recentFieldConvertor::setRecentFalseIfDuplicate);

        saveRecentFalseList(recentFalseListIfDuplicate);

        List<S> recentTrueList = StreamSupport.stream(entities.spliterator(), false)
                .filter(Objects::nonNull)
                .map(recentFieldConvertor::recentTrue)
                .filter(Objects::nonNull)
            .toList();

        return recentTrueList.stream()
                    .map(doc-> openSearchOperations.save(doc, rollingIndexName()))
                    .collect(toList());

    }

    private <S> void saveRecentFalseList( Map<String, List<S>> recentFalseListMap) {
        recentFalseListMap
            .forEach((key, values) -> values.forEach(
                value-> openSearchOperations.save(value, IndexCoordinates.of(key))
            ));
    }


    @Override
    public List<T> findRecentDocsByScrollApi(Query query) {

        OpenSearchRestTemplate template = (OpenSearchRestTemplate) openSearchOperations;
        IndexCoordinates indexCoordinatesFor = template.getIndexCoordinatesFor(entityClazz);

        SearchScrollHits<T> scroll
            = template.searchScrollStart(60000, query, entityClazz, indexCoordinatesFor);

        String scrollId = scroll.getScrollId();
        List<T> entities = new ArrayList<>();

        while (scroll.hasSearchHits()) {
            entities.addAll(
                scroll.getSearchHits()
                    .stream()
                    .map(SearchHit::getContent)
                    .toList()
            );
            scrollId = scroll.getScrollId();
            scroll = template.searchScrollContinue(scrollId, 60000, entityClazz,indexCoordinatesFor);
        }

        assert scrollId != null;
        template.searchScrollClear(List.of(scrollId));

        return entities;

    }

    @Override
    public List<T> findDocsByScrollApi(Query query) {

        NativeSearchQuery nativeSearchQuery = (NativeSearchQuery)query;

        OpenSearchRestTemplate template = (OpenSearchRestTemplate) openSearchOperations;

        IndexCoordinates indexCoordinatesFor = template.getIndexCoordinatesFor(entityClazz);

        SearchScrollHits<T> scroll
                = template.searchScrollStart(60000, nativeSearchQuery, entityClazz, indexCoordinatesFor);

        String scrollId = scroll.getScrollId();
        List<T> entities = new ArrayList<>();

        while (scroll.hasSearchHits()) {
            entities.addAll(
                    scroll.getSearchHits()
                            .stream()
                            .map(SearchHit::getContent)
                            .toList()
            );
            scrollId = scroll.getScrollId();
            scroll = template.searchScrollContinue(scrollId, 60000, entityClazz,indexCoordinatesFor);
        }

        assert scrollId != null;
        template.searchScrollClear(List.of(scrollId));

        return entities;

    }


    @Override
    public DocumentResultWrapper<T> findRecentDocsBySearchAfter(Query query, List<Object> searchAfter) {

        if(searchAfter==null){
            searchAfter = new ArrayList<>();
        }

        SearchHits<T> search = openSearchOperations.search(query, entityClazz);

        return new DocumentResultWrapper<>(search, entityClazz);
    }

    @Override
    public DocumentResultWrapper<T> findDocsBySearchAfter(Query query, List<Object> searchAfter) {

        if(searchAfter==null){
            searchAfter = new ArrayList<>();
        }

        NativeSearchQuery nativeSearchQuery = (NativeSearchQuery)query;

        if(!searchAfter.isEmpty()){
            nativeSearchQuery.setSearchAfter(searchAfter);
        }

        SearchHits<T> search = openSearchOperations.search(nativeSearchQuery, entityClazz);

        return new DocumentResultWrapper<>(search, entityClazz);
    }

    @Override
    public String indexAliasName() {
        IndexCoordinates indexCoordinatesFor = openSearchOperations.getIndexCoordinatesFor(entityClazz);
        return indexCoordinatesFor.getIndexName();
    }

    @Override
    public void deleteIndexWithDayRange(int day){
        this.indexWithDayRange(day)
            .forEach(indexName-> {
                    log.info("delete-index->{}",indexName);
                    this.deleteIndex(IndexCoordinates.of(indexName));
                }
            );
    }

    @Override
    public String deleteDocByEntity(T t) {
        return openSearchOperations.delete(t);
    }

    @Override
    public U deleteRecentDocById(U u) {
        this.deleteAllById(List.of(u));
        return u;
    }

    private IndexCoordinates rollingIndexName() {
        Document document = AnnotationUtils.findAnnotation(entityClazz, Document.class);
        if(document!=null){

            IndexCoordinates indexCoordinatesFor = openSearchOperations.getIndexCoordinatesFor(entityClazz);

            String indexName = indexCoordinatesFor.getIndexName();

            try{

                Method method = methodInfo(entityClazz, RollingIndexName.class);

                if(method!=null){
                    Constructor<T> constructor = entityClazz.getConstructor();
                    T t = constructor.newInstance();
                    return IndexCoordinates.of(indexName +"-"+method.invoke(t));

                }else{
                    return IndexCoordinates.of(indexName);
                }
            }catch (Exception e) {
                return IndexCoordinates.of(indexName);
            }
        }
        throw new IllegalArgumentException("인덱스명을 확인해주시길 바랍니다.");
    }

    private String rollingIndexNameSuffix() {
        String indexName = this.rollingIndexName().getIndexName();
        return Optional.of(indexName.split("-", 2))
                .filter(a -> a.length > 1)
                .map(a -> a[1]).orElse("0000-00-00");
    }

    @Override
    public void mergeWithReindex(int day) {

        Map<String, String[]> indexNameGroupByMonth = indexWithDayRange(day)
                .stream()
                .collect(Collectors.groupingBy(str -> str.substring(0, str.length()-2) + "00")).entrySet().stream()
                .collect(Collectors.toMap(
                        Map.Entry::getKey,
                        entry -> entry.getValue().toArray(new String[0])
                ));

        ReindexRequest reindexRequest = new ReindexRequest();
        reindexRequest.setTimeout(TimeValue.MAX_VALUE);
        reindexRequest.setRefresh(true);

        log.info("reindex 시작");
        indexNameGroupByMonth.forEach((key, value) -> {
           for (String docIndexName : value) {
               try {
                   reindexRequest.setSourceIndices(docIndexName);
                   reindexRequest.setDestIndex(key);
                   reindexRequest(reindexRequest, 0);

               } catch (Exception e) {
                   throw new IllegalArgumentException(e);
               }
           }
        });
        log.info("reindex 종료");

        log.info("reindex 이후 delete 시작");
        indexNameGroupByMonth.forEach((key, value) -> {
           for (String docIndexName : value) {
               try {
                   this.deleteIndex(IndexCoordinates.of(docIndexName));
               } catch (Exception e) {
                   throw new IllegalArgumentException(e);
               }
           }
        });
        log.info("reindex 이후 delete 종료");
    }

    private void reindexRequest(ReindexRequest reindexRequest, int count){
        try {
            RestHighLevelClient restHighLevelClient = getBean(RestHighLevelClient.class);
            restHighLevelClient.reindex(reindexRequest, RequestOptions.DEFAULT);
        } catch (Exception e) {
            try {
                Thread.sleep(10000);
            } catch (InterruptedException ex) {
                Thread.currentThread().interrupt();
            }
            if(count>9){
                log.warn("인덱스 머지 요청 시도 처리 초과");
                log.warn("인덱스 요청 본문 : {}" , reindexRequest);
                throw new IllegalArgumentException(e.getMessage());
            }
            count = count + 1 ;
            log.info("요청 재시도 : {}" , count);
            log.info("인덱스 요청 본문 : {}" , reindexRequest);
            reindexRequest(reindexRequest,count);

        }
    }

    @Override
    public List<CatIndexVO> catIndexVOList(){
        IndexCoordinates indexCoordinatesFor = openSearchOperations.getIndexCoordinatesFor(entityClazz);
        String indexName = indexCoordinatesFor.getIndexName();
        RestHighLevelClient restHighLevelClient = getBean(RestHighLevelClient.class);
        try{
            ObjectMapper objectMapper = new ObjectMapper();
            Response response = restHighLevelClient.getLowLevelClient().performRequest(
                new Request("GET", "/_cat/indices/"+indexName+"?format=json")
            );
            String responseBody = EntityUtils.toString(response.getEntity());
            return objectMapper.readValue(responseBody, new TypeReference<>() {});
        }catch (Exception e){
            throw new IllegalArgumentException(e.getMessage());
        }

    }

    @Override
    public Long indexCount(){
        RestHighLevelClient restHighLevelClient = getBean(RestHighLevelClient.class);
        try{
            ObjectMapper objectMapper = new ObjectMapper();
            Response response = restHighLevelClient.getLowLevelClient().performRequest(
                    new Request("GET", "/_alias/"+this.indexAliasName())
            );
            String responseBody = EntityUtils.toString(response.getEntity());
            return (long)objectMapper.readValue(responseBody, HashMap.class).size();

        }catch (Exception e){
            throw new IllegalArgumentException(e.getMessage());
        }
    }

    private Set<String> indexNames() {
        IndexCoordinates indexCoordinatesFor = openSearchOperations.getIndexCoordinatesFor(entityClazz);
        String indexName = indexCoordinatesFor.getIndexName();
        IndexOperations indexOperations = openSearchOperations.indexOps(IndexCoordinates.of(indexName));
        return indexOperations.getAliasesForIndex(indexName).keySet();
    }

    private List<String> indexWithDayRange(int day){
        Set<String> indexNames = this.indexNames();
        List<String> days = dayList(day);
        return indexNames.stream()
            .filter(indexName -> !days.contains(indexName)
                && indexName.split("-",2)[1].replace("-", "").lastIndexOf("00") != 6)
            .collect(toList());
    }

    private void deleteIndex(IndexCoordinates indexCoordinates) {
        IndexOperations indexOperations = openSearchOperations.indexOps(indexCoordinates);
        indexOperations.delete();
    }

    private String findDateFormat(String date){

        String[] formats = {"yyyy-MM-dd", "yyyyMMdd"};

        for (String format : formats) {
            SimpleDateFormat sdf = new SimpleDateFormat(format);
            sdf.setLenient(false);
            try {
                sdf.parse(date);
                return format;
            }  catch (ParseException ignored) {
                return formats[0];
            }
        }
        return formats[0];

    }

    private List<String> dayList(int range) {

        Calendar calendar = Calendar.getInstance();
        calendar.setTime(new Date());

        calendar.add(Calendar.DATE, -(Math.abs(range) - 1));
        Date daysAgo = calendar.getTime();

        calendar.setTime(daysAgo);

        return IntStream.range(0, range).boxed().map(a -> {
            String day = defaultFormater.format(calendar.getTime());
            calendar.add(Calendar.DATE, 1);
            return this.indexAliasName()+"-"+ day;
        }).collect(Collectors.toList());

    }

}
