package com.arms.egovframework.javaservice.esframework.model.vo;

import static java.util.stream.Collectors.*;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;
import java.util.stream.Stream;

import lombok.Getter;
import org.opensearch.search.aggregations.Aggregation; // Changed import
import org.opensearch.search.aggregations.Aggregations; // Changed import
import org.opensearch.search.aggregations.bucket.MultiBucketsAggregation; // Changed import
import org.opensearch.search.aggregations.metrics.NumericMetricsAggregation;
import org.springframework.data.elasticsearch.core.AggregationsContainer;
import org.springframework.data.elasticsearch.core.SearchHits;


public class DocumentAggregations implements SearchDocResult {

	@Getter
	private final Long totalHits;

	@Getter
	private final Long docSumCount;

	private final List<DocumentBucket> documentBuckets;

	public DocumentAggregations(SearchHits<?> searchHits) {
		this.totalHits = searchHits.getTotalHits();

		AggregationsContainer<?> aggregations = searchHits.getAggregations();
		Map<String, Aggregation> mainBucketsAggregationMap = this.getTerm(aggregations)
				.entrySet()
				.stream()
				.collect(toMap(Entry::getKey, Entry::getValue));

		this.documentBuckets
				= new DocumentAggregations(mainBucketsAggregationMap).docBuckets();

		this.docSumCount = documentBuckets.stream().mapToLong(DocumentBucket::getDocCount).reduce(0, Long::sum);

	}

	public DocumentAggregations(List<DocumentBucket> documentBuckets) {
		this.totalHits = -1L;
		this.docSumCount = -1L;
		this.documentBuckets = documentBuckets;
	}

	DocumentAggregations(Map<String, Aggregation> bucketsAggregationMap) {

		this.documentBuckets = bucketsAggregationMap.entrySet()
				.stream()
				.flatMap(this::createDocumentBucket).collect(toList());
		this.totalHits = -1L;
		this.docSumCount = -1L;
	}

	private Stream<DocumentBucket> createDocumentBucket(Entry<String, Aggregation> bucket) {

		if(bucket.getValue() instanceof NumericMetricsAggregation.SingleValue) {
			return Stream.of((NumericMetricsAggregation.SingleValue)bucket.getValue())
					.map(bucketValue -> new DocumentBucket(bucket.getKey(), bucketValue));
		}

		return ((MultiBucketsAggregation) bucket.getValue())
				.getBuckets()
				.stream().map(bucketValue -> new DocumentBucket(bucket.getKey(),bucketValue));

	}

	private Map<String, Aggregation> getTerm(AggregationsContainer<?> aggregationsContainer){

		 return Optional.ofNullable(aggregationsContainer)
				 .map(a->((Aggregations)a.aggregations()).getAsMap())
				 .orElseGet(
				HashMap::new);
	}


	public List<DocumentBucket> deepestList() {
		return documentBuckets.stream()
				.flatMap(a -> a.getDeepestDocument().stream())
				.collect(toList());
	}


	public List<DocumentBucket> docBuckets() {
		return documentBuckets;
	}

}