package com.arms.api.report.export_service.generator;

import com.arms.api.report.export_service.config.ReportConfig;
import com.arms.api.report.export_service.exception.ReportExportException;
import com.arms.api.report.export_service.model.ChartData;
import com.arms.api.report.export_service.model.ReportData;
import com.arms.api.report.export_service.model.ReportSection;
import com.arms.api.report.export_service.model.TableData;
import com.arms.api.report.export_service.template.PlaceholderResolver;
import com.arms.api.report.export_service.template.TemplateLoader;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.poi.sl.usermodel.PictureData;
import org.apache.poi.xslf.usermodel.*;
import org.springframework.stereotype.Component;

import java.awt.*;
import java.awt.geom.Rectangle2D;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.List;
import java.util.Map;

/**
 * PPT 문서 생성기
 *
 * <p>Apache POI를 사용하여 PowerPoint 프레젠테이션을 생성합니다.
 *
 * <p>주요 기능:
 * <ul>
 *   <li>템플릿 로딩 및 플레이스홀더 치환</li>
 *   <li>동적 슬라이드 생성 (텍스트, 테이블, 차트)</li>
 *   <li>이미지 삽입</li>
 * </ul>
 *
 * @author HS.Yang
 * @since 25.12.03
 */
@Slf4j
@Component
@RequiredArgsConstructor
public class PptGenerator {

    private final TemplateLoader templateLoader;
    private final PlaceholderResolver placeholderResolver;
    private final ReportConfig reportConfig;
    private final ChartImageGenerator chartImageGenerator;

    /**
     * ReportData를 기반으로 PPT 생성
     *
     * @param data 리포트 데이터
     * @return 생성된 PPT 바이트 배열
     */
    public byte[] generate(ReportData data) {
        try {
            // 1. 템플릿 로드 또는 새 프레젠테이션 생성
            XMLSlideShow ppt = loadOrCreatePresentation(data);

            // 2. 플레이스홀더 치환
            Map<String, String> placeholders = data.buildDefaultPlaceholders();
            placeholderResolver.resolveAllSlides(ppt, placeholders);

            // 3. 섹션별 슬라이드 추가
            for (ReportSection section : data.getSections()) {
                addSectionSlide(ppt, section, data);
            }

            // 4. 테이블 슬라이드 추가
            for (TableData table : data.getTables()) {
                addTableSlide(ppt, table);
            }

            // 5. 차트 슬라이드 추가
            for (ChartData chart : data.getCharts()) {
                addChartSlide(ppt, chart);
            }

            // 6. 바이트 배열로 변환
            return writeToByteArray(ppt);

        } catch (IOException e) {
            log.error("Failed to generate PPT", e);
            throw new ReportExportException("PPT_GENERATION_ERROR", "Failed to generate PPT", e);
        }
    }

    /**
     * 템플릿 로드 또는 새 프레젠테이션 생성
     */
    private XMLSlideShow loadOrCreatePresentation(ReportData data) {
        if (data.getTemplateName() != null && templateLoader.templateExists(data.getTemplateName())) {
            log.info("Loading template: {}", data.getTemplateName());
            return templateLoader.loadPptTemplate(data.getTemplateName());
        }

        log.info("Creating default presentation (no template)");
        return createDefaultPresentation(data);
    }

    /**
     * 기본 프레젠테이션 생성 (템플릿 없이)
     */
    private XMLSlideShow createDefaultPresentation(ReportData data) {
        XMLSlideShow ppt = new XMLSlideShow();

        XSLFSlideMaster master = ppt.getSlideMasters().get(0);
        XSLFSlideLayout titleLayout = master.getLayout(SlideLayout.TITLE);
        XSLFSlide titleSlide = ppt.createSlide(titleLayout);

        // 제목 설정
        XSLFTextShape title = titleSlide.getPlaceholder(0);
        if (title != null) {
            title.setText(data.getTitle());
            setTextStyle(title, reportConfig.getPpt().getTitleFontSize(), true);
        }

        // 부제목 설정
        XSLFTextShape subtitle = titleSlide.getPlaceholder(1);
        if (subtitle != null && data.getSubtitle() != null) {
            subtitle.setText(data.getSubtitle());
        }

        return ppt;
    }

    /**
     * 섹션 슬라이드 추가
     */
    private void addSectionSlide(XMLSlideShow ppt, ReportSection section, ReportData data) {
        XSLFSlideMaster master = ppt.getSlideMasters().get(0);
        XSLFSlideLayout contentLayout = master.getLayout(SlideLayout.TITLE_AND_CONTENT);
        XSLFSlide slide = ppt.createSlide(contentLayout);

        // 제목 설정
        XSLFTextShape title = slide.getPlaceholder(0);
        if (title != null) {
            title.setText(section.getTitle());
        }

        // 내용 설정
        XSLFTextShape content = slide.getPlaceholder(1);
        if (content != null) {
            content.clearText();

            switch (section.getType()) {
                case BULLET_LIST:
                    addBulletPoints(content, section.getBulletPoints());
                    break;
                case TEXT:
                    addTextContent(content, section.getContent());
                    break;
                case TABLE:
                    addTableReference(slide, section, data);
                    break;
                case CHART:
                    addChartReference(slide, section, data);
                    break;
                default:
                    addTextContent(content, section.getContent());
            }
        }

        // 이미지 추가
        if (section.hasImage()) {
            addImageToSlide(ppt, slide, section.getImage(), section.getImageCaption());
        }
    }

    /**
     * 불릿 포인트 추가
     */
    private void addBulletPoints(XSLFTextShape textShape, List<String> bulletPoints) {
        if (bulletPoints == null || bulletPoints.isEmpty()) {
            return;
        }

        for (int i = 0; i < bulletPoints.size(); i++) {
            XSLFTextParagraph paragraph;
            List<XSLFTextParagraph> paragraphs = textShape.getTextParagraphs();
            if (i == 0 && !paragraphs.isEmpty()) {
                paragraph = paragraphs.get(0);
            } else {
                paragraph = textShape.addNewTextParagraph();
            }
            paragraph.setBullet(true);
            XSLFTextRun run = paragraph.addNewTextRun();
            run.setText(bulletPoints.get(i));
            run.setFontSize((double) reportConfig.getPpt().getBodyFontSize());
        }
    }

    /**
     * 텍스트 내용 추가
     */
    private void addTextContent(XSLFTextShape textShape, String content) {
        if (content == null) {
            return;
        }

        List<XSLFTextParagraph> paragraphs = textShape.getTextParagraphs();
        XSLFTextParagraph paragraph;
        if (paragraphs.isEmpty()) {
            paragraph = textShape.addNewTextParagraph();
        } else {
            paragraph = paragraphs.get(0);
        }
        paragraph.setBullet(false);
        XSLFTextRun run = paragraph.addNewTextRun();
        run.setText(content);
        run.setFontSize((double) reportConfig.getPpt().getBodyFontSize());
    }

    /**
     * 테이블 슬라이드 추가
     */
    private void addTableSlide(XMLSlideShow ppt, TableData tableData) {
        XSLFSlideMaster master = ppt.getSlideMasters().get(0);
        XSLFSlideLayout blankLayout = master.getLayout(SlideLayout.BLANK);
        XSLFSlide slide = ppt.createSlide(blankLayout);

        // 테이블 제목
        if (tableData.getTableName() != null) {
            XSLFTextBox titleBox = slide.createTextBox();
            titleBox.setAnchor(new Rectangle2D.Double(50, 30, 620, 40));
            XSLFTextParagraph p = titleBox.addNewTextParagraph();
            XSLFTextRun r = p.addNewTextRun();
            r.setText(tableData.getTableName());
            r.setFontSize(20.0);
            r.setBold(true);
        }

        // 테이블 생성
        int numRows = tableData.getRowCount() + 1;
        int numCols = tableData.getColumnCount();

        if (numCols == 0) {
            log.warn("Table has no columns, skipping");
            return;
        }

        XSLFTable table = slide.createTable(numRows, numCols);
        table.setAnchor(new Rectangle2D.Double(50, 80, 620, 400));

        // 열 너비 설정
        double colWidth = 620.0 / numCols;
        for (int i = 0; i < numCols; i++) {
            table.setColumnWidth(i, colWidth);
        }

        // 헤더 행
        XSLFTableRow headerRow = table.getRows().get(0);
        for (int col = 0; col < numCols; col++) {
            XSLFTableCell cell = headerRow.getCells().get(col);
            cell.setText(tableData.getHeaders().get(col));
            cell.setFillColor(Color.decode(tableData.getStyle().getHeaderBackgroundColor()));
            setCellTextStyle(cell, true, Color.WHITE);
        }

        // 데이터 행
        for (int row = 0; row < tableData.getRowCount(); row++) {
            XSLFTableRow dataRow = table.getRows().get(row + 1);
            List<String> rowData = tableData.getRows().get(row);

            for (int col = 0; col < numCols; col++) {
                XSLFTableCell cell = dataRow.getCells().get(col);
                String cellText = col < rowData.size() ? rowData.get(col) : "";
                cell.setText(cellText);

                if (tableData.getStyle().isHasAlternateRowColor() && row % 2 == 1) {
                    cell.setFillColor(Color.decode(tableData.getStyle().getAlternateRowColor()));
                }
                setCellTextStyle(cell, false, Color.BLACK);
            }
        }
    }

    /**
     * 차트 슬라이드 추가
     */
    private void addChartSlide(XMLSlideShow ppt, ChartData chartData) throws IOException {
        XSLFSlideMaster master = ppt.getSlideMasters().get(0);
        XSLFSlideLayout blankLayout = master.getLayout(SlideLayout.BLANK);
        XSLFSlide slide = ppt.createSlide(blankLayout);

        // 차트 제목
        if (chartData.getTitle() != null) {
            XSLFTextBox titleBox = slide.createTextBox();
            titleBox.setAnchor(new Rectangle2D.Double(50, 30, 620, 40));
            XSLFTextParagraph p = titleBox.addNewTextParagraph();
            XSLFTextRun r = p.addNewTextRun();
            r.setText(chartData.getTitle());
            r.setFontSize(20.0);
            r.setBold(true);
        }

        // 차트 이미지 생성
        byte[] chartImage;
        if (chartData.hasRenderedImage()) {
            chartImage = chartData.getRenderedImage();
        } else {
            chartImage = chartImageGenerator.generateChartImage(chartData);
        }

        // 이미지 추가
        addImageToSlide(ppt, slide, chartImage, null, 60, 80, 600, 420);
    }

    /**
     * 이미지 추가 (기본 위치)
     */
    private void addImageToSlide(XMLSlideShow ppt, XSLFSlide slide,
                                byte[] imageData, String caption) {
        addImageToSlide(ppt, slide, imageData, caption, 150, 150, 420, 300);
    }

    /**
     * 이미지 추가 (위치 지정)
     */
    private void addImageToSlide(XMLSlideShow ppt, XSLFSlide slide,
                                byte[] imageData, String caption,
                                double x, double y, double width, double height) {
        try {
            XSLFPictureData pictureData = ppt.addPicture(imageData, PictureData.PictureType.PNG);
            XSLFPictureShape picture = slide.createPicture(pictureData);
            picture.setAnchor(new Rectangle2D.Double(x, y, width, height));

            if (caption != null && !caption.isEmpty()) {
                XSLFTextBox captionBox = slide.createTextBox();
                captionBox.setAnchor(new Rectangle2D.Double(x, y + height + 10, width, 30));
                XSLFTextParagraph p = captionBox.addNewTextParagraph();
                XSLFTextRun r = p.addNewTextRun();
                r.setText(caption);
                r.setFontSize(10.0);
                r.setItalic(true);
            }
        } catch (Exception e) {
            log.error("Failed to add image to slide", e);
        }
    }

    /**
     * 테이블 참조 처리
     */
    private void addTableReference(XSLFSlide slide, ReportSection section, ReportData data) {
        for (TableData table : data.getTables()) {
            if (table.getTableId() != null && table.getTableId().equals(section.getTableRef())) {
                XSLFTextShape content = slide.getPlaceholder(1);
                if (content != null) {
                    content.setText("See table: " + table.getTableName());
                }
                break;
            }
        }
    }

    /**
     * 차트 참조 처리
     */
    private void addChartReference(XSLFSlide slide, ReportSection section, ReportData data) {
        for (ChartData chart : data.getCharts()) {
            if (chart.getChartId() != null && chart.getChartId().equals(section.getChartRef())) {
                XSLFTextShape content = slide.getPlaceholder(1);
                if (content != null) {
                    content.setText("See chart: " + chart.getTitle());
                }
                break;
            }
        }
    }

    /**
     * 텍스트 스타일 설정
     */
    private void setTextStyle(XSLFTextShape textShape, int fontSize, boolean bold) {
        for (XSLFTextParagraph p : textShape.getTextParagraphs()) {
            for (XSLFTextRun r : p.getTextRuns()) {
                r.setFontSize((double) fontSize);
                r.setBold(bold);
            }
        }
    }

    /**
     * 셀 텍스트 스타일 설정
     */
    private void setCellTextStyle(XSLFTableCell cell, boolean bold, Color fontColor) {
        for (XSLFTextParagraph p : cell.getTextParagraphs()) {
            for (XSLFTextRun r : p.getTextRuns()) {
                r.setFontSize((double) reportConfig.getPpt().getTableFontSize());
                r.setBold(bold);
                r.setFontColor(fontColor);
            }
        }
    }

    /**
     * 바이트 배열로 변환
     */
    private byte[] writeToByteArray(XMLSlideShow ppt) throws IOException {
        try (ByteArrayOutputStream out = new ByteArrayOutputStream()) {
            ppt.write(out);
            return out.toByteArray();
        } finally {
            ppt.close();
        }
    }
}
