package com.arms.api.report.export_service.exception;

import com.arms.api.report.export_service.dto.ReportExportResponse;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.RestControllerAdvice;

/**
 * 리포트 전역 예외 핸들러
 *
 * @author HS.Yang
 * @since 25.12.03
 */
@Slf4j
@RestControllerAdvice(basePackages = "com.arms.api.report.export_service")
public class ReportExceptionHandler {

    @ExceptionHandler(TemplateNotFoundException.class)
    public ResponseEntity<ReportExportResponse> handleTemplateNotFound(TemplateNotFoundException e) {
        log.error("Template not found: {}", e.getTemplateName(), e);
        return ResponseEntity
                .status(HttpStatus.NOT_FOUND)
                .body(ReportExportResponse.failure(e.getErrorCode(), e.getDetail()));
    }

    @ExceptionHandler(ReportExportException.class)
    public ResponseEntity<ReportExportResponse> handleReportExportException(ReportExportException e) {
        log.error("Report export error: {}", e.getMessage(), e);
        return ResponseEntity
                .status(HttpStatus.INTERNAL_SERVER_ERROR)
                .body(ReportExportResponse.failure(e.getErrorCode(), e.getDetail()));
    }

    @ExceptionHandler(MethodArgumentNotValidException.class)
    public ResponseEntity<ReportExportResponse> handleValidationException(MethodArgumentNotValidException e) {
        log.error("Validation error: {}", e.getMessage());
        String errorMessage = e.getBindingResult().getFieldErrors().stream()
                .findFirst()
                .map(error -> error.getField() + ": " + error.getDefaultMessage())
                .orElse("Validation failed");
        return ResponseEntity
                .status(HttpStatus.BAD_REQUEST)
                .body(ReportExportResponse.failure("VALIDATION_ERROR", errorMessage));
    }

    @ExceptionHandler(Exception.class)
    public ResponseEntity<ReportExportResponse> handleGenericException(Exception e) {
        log.error("Unexpected error during report export", e);
        return ResponseEntity
                .status(HttpStatus.INTERNAL_SERVER_ERROR)
                .body(ReportExportResponse.failure("UNEXPECTED_ERROR", e.getMessage()));
    }
}