package com.arms.api.report.export_service.dto;

import com.arms.api.report.export_service.model.ChartData;
import com.arms.api.report.export_service.model.ReportData;
import com.arms.api.report.export_service.model.ReportSection;
import com.arms.api.report.export_service.model.TableData;
import lombok.*;

import javax.validation.constraints.NotBlank;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * 리포트 내보내기 요청 DTO
 *
 * @author HS.Yang
 * @since 25.12.03
 */
@Getter
@Setter
@Builder
@NoArgsConstructor
@AllArgsConstructor
public class ReportExportRequest {

    /** 리포트 제목 (필수) */
    @NotBlank(message = "Title is required")
    private String title;

    /** 부제목 */
    private String subtitle;

    /** 설명 */
    private String description;

    /** 작성자 */
    private String author;

    /** 부서 */
    private String department;

    /** 템플릿 이름 */
    private String templateName;

    /** 섹션 목록 */
    private List<SectionRequest> sections;

    /** 테이블 목록 */
    private List<TableRequest> tables;

    /** 차트 목록 */
    private List<ChartRequest> charts;

    /** 커스텀 플레이스홀더 */
    private Map<String, String> customPlaceholders;

    @Getter
    @Setter
    @NoArgsConstructor
    @AllArgsConstructor
    public static class SectionRequest {
        private String title;
        private String content;
        private String type;
        private int order;
        private List<String> bulletPoints;
        private String tableRef;
        private String chartRef;
    }

    @Getter
    @Setter
    @NoArgsConstructor
    @AllArgsConstructor
    public static class TableRequest {
        private String tableId;
        private String tableName;
        private String caption;
        private List<String> headers;
        private List<List<String>> rows;
    }

    @Getter
    @Setter
    @NoArgsConstructor
    @AllArgsConstructor
    public static class ChartRequest {
        private String chartId;
        private String title;
        private String type;
        private List<String> labels;
        private List<SeriesRequest> series;
    }

    @Getter
    @Setter
    @NoArgsConstructor
    @AllArgsConstructor
    public static class SeriesRequest {
        private String name;
        private List<Double> values;
        private String color;
    }

    /**
     * 요청 DTO를 ReportData 모델로 변환
     */
    public ReportData toReportData() {
        ReportData.ReportDataBuilder builder = ReportData.builder()
                .title(this.title)
                .subtitle(this.subtitle)
                .description(this.description)
                .author(this.author)
                .department(this.department)
                .templateName(this.templateName)
                .reportDate(LocalDateTime.now());

        // 섹션 변환
        if (sections != null && !sections.isEmpty()) {
            List<ReportSection> reportSections = new ArrayList<>();
            for (SectionRequest section : sections) {
                reportSections.add(convertSection(section));
            }
            builder.sections(reportSections);
        }

        // 테이블 변환
        if (tables != null && !tables.isEmpty()) {
            List<TableData> tableDataList = new ArrayList<>();
            for (TableRequest table : tables) {
                tableDataList.add(convertTable(table));
            }
            builder.tables(tableDataList);
        }

        // 차트 변환
        if (charts != null && !charts.isEmpty()) {
            List<ChartData> chartDataList = new ArrayList<>();
            for (ChartRequest chart : charts) {
                chartDataList.add(convertChart(chart));
            }
            builder.charts(chartDataList);
        }

        if (customPlaceholders != null) {
            builder.placeholders(customPlaceholders);
        }

        return builder.build();
    }

    private ReportSection convertSection(SectionRequest request) {
        ReportSection.SectionType sectionType = ReportSection.SectionType.TEXT;
        if (request.getType() != null && !request.getType().isEmpty()) {
            try {
                sectionType = ReportSection.SectionType.valueOf(request.getType().toUpperCase());
            } catch (IllegalArgumentException e) {
                sectionType = ReportSection.SectionType.TEXT;
            }
        }

        return ReportSection.builder()
                .title(request.getTitle())
                .content(request.getContent())
                .type(sectionType)
                .order(request.getOrder())
                .bulletPoints(request.getBulletPoints() != null ?
                        request.getBulletPoints() : new ArrayList<>())
                .tableRef(request.getTableRef())
                .chartRef(request.getChartRef())
                .build();
    }

    private TableData convertTable(TableRequest request) {
        return TableData.builder()
                .tableId(request.getTableId())
                .tableName(request.getTableName())
                .caption(request.getCaption())
                .headers(request.getHeaders() != null ?
                        request.getHeaders() : new ArrayList<>())
                .rows(request.getRows() != null ?
                        request.getRows() : new ArrayList<>())
                .build();
    }

    // convert request to ChartData
    private ChartData convertChart(ChartRequest request) {
        ChartData.ChartType chartType = ChartData.ChartType.BAR;
        if (request.getType() != null && !request.getType().isEmpty()) {
            try {
                chartType = ChartData.ChartType.valueOf(request.getType().toUpperCase());
            } catch (IllegalArgumentException e) {
                chartType = ChartData.ChartType.BAR;
            }
        }

        ChartData.ChartDataBuilder builder = ChartData.builder()
                .chartId(request.getChartId())
                .title(request.getTitle())
                .type(chartType)
                .labels(request.getLabels() != null ?
                        request.getLabels() : new ArrayList<>());

        if (request.getSeries() != null) {
            List<ChartData.DataSeries> seriesList = new ArrayList<>();
            for (SeriesRequest s : request.getSeries()) {
                seriesList.add(ChartData.DataSeries.builder()
                        .name(s.getName())
                        .values(s.getValues() != null ? s.getValues() : new ArrayList<>())
                        .color(s.getColor())
                        .build());
            }
            builder.series(seriesList);
        }

        return builder.build();
    }
}
