package com.arms.api.poc.service;

import com.arms.api.poc.model.PocDTO;
import com.arms.api.poc.model.PocEntity;
import com.arms.api.poc.model.vo.PocVO;
import com.arms.api.report.mail.modal.entity.sender.SenderEntity;
import com.arms.api.report.mail.service.sender.SenderService;
import com.arms.api.util.communicate.external.EngineService;
import com.arms.api.util.slack.SlackNotificationService;
import com.arms.api.util.slack.SlackProperty;
import com.arms.egovframework.javaservice.treeframework.controller.CommonResponse;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.ResponseEntity;
import org.springframework.mail.javamail.JavaMailSenderImpl;
import org.springframework.mail.javamail.MimeMessageHelper;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;
import org.thymeleaf.spring5.SpringTemplateEngine;
import org.thymeleaf.context.Context;

import javax.mail.internet.MimeMessage;
import java.util.Properties;
import java.util.List;
import java.util.Set;

@Slf4j
@Service("pocService")
@RequiredArgsConstructor
public class PocServiceImpl implements PocService {

    private final SenderService senderService;
    private final SpringTemplateEngine templateEngine;
    private final SlackNotificationService slackNotificationService;
    private final EngineService engineService;

    private static final Set<String> LAST_NAME_FIRST_COUNTRIES = Set.of("KR", "JP", "CN");

    @Override
    public PocEntity addPoc(PocDTO pocDTO) throws Exception {

        pocDTO.setC_poc_full_name(formatFullName(pocDTO));
        pocDTO.setC_poc_status(pocDTO.getC_poc_status() != null ? pocDTO.getC_poc_status() : "PENDING");
        pocDTO.setC_poc_assignee_id(null);

        ResponseEntity<CommonResponse.ApiResult<String>> response = engineService.createPoc(pocDTO);
        String pocId = response.getBody().getResponse();

        PocEntity savedEntity = engineService.getPoc(pocId).getBody().getResponse();

        sendPocEmailNotification(savedEntity);
        sendPocSuccessEmailToApplicant(savedEntity);
        sendPocSlackNotification(savedEntity);

        return savedEntity;
    }

    @Override
    public PocEntity getPoc(String id) throws Exception {
        ResponseEntity<CommonResponse.ApiResult<PocEntity>> response = engineService.getPoc(id);
        return response.getBody().getResponse();
    }

    @Override
    public PocVO getPocList(PocDTO pocDTO) throws Exception {
        ResponseEntity<CommonResponse.ApiResult<PocVO>> response = engineService.getPocList(pocDTO);
        return response.getBody().getResponse();
    }

    @Override
    public int updatePoc(PocDTO pocDTO) throws Exception {

        if (pocDTO.getId() == null) {
            throw new IllegalArgumentException("PoC ID 누락");
        }

        ResponseEntity<CommonResponse.ApiResult<String>> response = engineService.updatePoc(pocDTO.getId(), pocDTO);

        PocEntity pocEntity = engineService.getPoc(pocDTO.getId()).getBody().getResponse();

        sendPocSuccessEmailToAssignee(pocEntity);
        sendPocSlackToAssignee(pocEntity);

        return response.getBody().getResponse() != null ? 1 : 0;
    }

    @Override
    public int removePoc(String id) throws Exception {
        ResponseEntity<CommonResponse.ApiResult<String>> response = engineService.deletePoc(id);
        return response.getBody().getResponse() != null ? 1 : 0;
    }

    /**
     * 국가별 이름 표기 순서에 따라 전체 이름을 포맷팅
     * 예시:
     * - 한국(KR): 이보석 (성+공백+이름, 공백 없음)
     * - 영미권/유럽: Boseok Lee (이름+공백+성)
     */
    private String formatFullName(PocDTO pocDTO){
        String firstName = pocDTO.getC_poc_first_name() != null ? pocDTO.getC_poc_first_name() : "";
        String lastName = pocDTO.getC_poc_last_name() != null ? pocDTO.getC_poc_last_name() : "";

        if(LAST_NAME_FIRST_COUNTRIES.contains(pocDTO.getC_poc_country())){
            return lastName + " " + firstName;
        }else{
            return firstName + " " + lastName;
        }
    }


    @Async("taskExecutor-arms")
    private void sendPocEmailNotification(PocEntity pocEntity) throws Exception {

        SenderEntity smtpConfig = getSenderConfig();
        JavaMailSenderImpl mailSender = createMailSender(smtpConfig);

        Context context = new Context();
        context.setVariable("pocEntity", pocEntity);
        String htmlContent = templateEngine.process("poc/poc-notification-mail", context);

        MimeMessage mimeMessage = mailSender.createMimeMessage();
        MimeMessageHelper helper = new MimeMessageHelper(mimeMessage, true, "UTF-8");

        helper.setTo("313@313.co.kr");
        helper.setFrom(smtpConfig.getC_sender_email());
        helper.setSubject("[A-RMS] 새로운 POC 신청 - " + pocEntity.getC_poc_company_name());
        helper.setText(htmlContent, true);

        mailSender.send(mimeMessage);
    }

    @Async("taskExecutor-arms")
    private void sendPocSuccessEmailToAssignee(PocEntity pocEntity) throws Exception {

        SenderEntity smtpConfig = getSenderConfig();
        JavaMailSenderImpl mailSender = createMailSender(smtpConfig);

        Context context = new Context();
        context.setVariable("pocEntity", pocEntity);
        String htmlContent = templateEngine.process("poc/poc-assignee-mail", context);

        MimeMessage mimeMessage = mailSender.createMimeMessage();
        MimeMessageHelper helper = new MimeMessageHelper(mimeMessage, true, "UTF-8");

        String assigneeEmail = pocEntity.getC_poc_assignee_email();
        if (assigneeEmail == null || assigneeEmail.trim().isEmpty()) {
            assigneeEmail = "313@313.co.kr";
        }

        helper.setTo(assigneeEmail);
        helper.setFrom(smtpConfig.getC_sender_email());
        helper.setSubject("[A-RMS] 새로운 POC 신청 - " + pocEntity.getC_poc_company_name());
        helper.setText(htmlContent, true);

        mailSender.send(mimeMessage);
    }


    @Async("taskExecutor-arms")
    private void sendPocSuccessEmailToApplicant(PocEntity pocEntity) throws Exception {

        SenderEntity smtpConfig = getSenderConfig();
        JavaMailSenderImpl mailSender = createMailSender(smtpConfig);

        Context context = new Context();
        context.setVariable("pocEntity", pocEntity);
        context.setVariable("recipientName", pocEntity.getC_poc_full_name());
        context.setVariable("recipientCompany", pocEntity.getC_poc_company_name());
        context.setVariable("recipientEmail", pocEntity.getC_poc_email_address());

        String htmlContent = templateEngine.process("poc/poc-success-mail", context);

        MimeMessage mimeMessage = mailSender.createMimeMessage();
        MimeMessageHelper helper = new MimeMessageHelper(mimeMessage, true, "UTF-8");

        helper.setTo(pocEntity.getC_poc_email_address());
        helper.setFrom(smtpConfig.getC_sender_email());
        helper.setSubject("[A-RMS] POC 신청이 접수되었습니다");
        helper.setText(htmlContent, true);

        mailSender.send(mimeMessage);
    }

    @Async("taskExecutor-arms")
    private void sendPocSlackNotification(PocEntity pocEntity) throws Exception{

        Context context = new Context();
        context.setVariable("pocEntity", pocEntity);
        String slackMessage = templateEngine.process("poc/poc-notification-slack", context);

        slackNotificationService.sendMessageToChannel(SlackProperty.Channel.backend, slackMessage);
    }

    @Async("taskExecutor-arms")
    private void sendPocSlackToAssignee(PocEntity pocEntity) throws Exception{

        Context context = new Context();
        context.setVariable("pocEntity", pocEntity);
        String slackMessage = templateEngine.process("poc/poc-assignee-slack", context);

        slackNotificationService.sendMessageToChannel(SlackProperty.Channel.backend, slackMessage);
    }


    private SenderEntity getSenderConfig() throws Exception {
        List<SenderEntity> senderInfo = senderService.getNodesWithoutRoot(new SenderEntity());
        if (senderInfo == null || senderInfo.isEmpty()) {
            throw new RuntimeException("SMTP 설정을 찾을 수 없습니다.");
        }
        return senderInfo.get(0);
    }


    private JavaMailSenderImpl createMailSender(SenderEntity smtpConfig) {
        JavaMailSenderImpl mailSender = new JavaMailSenderImpl();

        mailSender.setHost(smtpConfig.getC_smtp_server());
        mailSender.setPort(smtpConfig.getC_smtp_port());
        mailSender.setUsername(smtpConfig.getC_sender_email());
        mailSender.setPassword(smtpConfig.getC_sender_password());

        Properties props = mailSender.getJavaMailProperties();
        props.put("mail.smtp.auth", "true");
        props.put("mail.smtp.ssl.enable", "true");
        props.put("mail.smtp.starttls.enable", "false");
        props.put("mail.smtp.ssl.protocols", "TLSv1.2");

        return mailSender;
    }
}
