package com.arms.api.analysis.cost.controller;

import com.arms.api.analysis.cost.model.vo.SalaryVO;
import com.arms.api.analysis.cost.model.vo.SalaryEntity;
import com.arms.api.analysis.cost.model.dto.SampleDTO;
import com.arms.api.analysis.cost.service.SalaryService;
import com.arms.egovframework.javaservice.treeframework.controller.CommonResponse;
import com.arms.egovframework.javaservice.treeframework.controller.TreeAbstractController;
import com.arms.egovframework.javaservice.treeframework.excel.ExcelUtilsBase;
import com.arms.egovframework.javaservice.treeframework.excel.ExcelUtilsFactory;
import io.swagger.annotations.ApiOperation;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.HttpHeaders;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.servlet.ModelAndView;

import javax.annotation.PostConstruct;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

@Slf4j
@RestController
@RequestMapping("/admin/arms/salaries")
@RequiredArgsConstructor
public class SalaryController extends TreeAbstractController<SalaryService, SalaryVO, SalaryEntity> {

    private final SalaryService salaryService;

    @PostConstruct
    public void initialize() {
        setTreeService(salaryService);
        setTreeEntity(SalaryEntity.class);
    }

    /**
     * 연봉 입력을 위한 엑셀 템플릿 다운로드 API
     */
    @ApiOperation(value = "연봉 입력 엑셀 템플릿 다운로드")
    @PostMapping("/excel-download.do")
    public void 연봉입력_엑셀템플릿_다운로드(@RequestBody List<SalaryVO> 연봉데이터리스트,
                                @RequestParam("excelFileName") String excelFileName,
                                HttpServletResponse httpServletResponse) throws Exception {

        log.info(" [ " + this.getClass().getName() + " :: 연봉입력_엑셀템플릿_다운로드 ]");

        httpServletResponse.addHeader(HttpHeaders.CONTENT_DISPOSITION, "attachment; filename=" + excelFileName);
        httpServletResponse.addHeader("Cache-Control", "no-cache, no-store, must-revalidate");
        httpServletResponse.addHeader("Pragma", "no-cache");
        httpServletResponse.addHeader("Expires", "0");
        httpServletResponse.setContentType("application/octet-stream");

        ExcelUtilsBase excelUtilsBase = ExcelUtilsFactory.getInstance(httpServletResponse.getOutputStream());
        List<SampleDTO> 샘플데이터 = salaryService.getSampleSalaryInfo();
        List<SalaryEntity> 비교한_연봉리스트 = salaryService.compareSalaryInfo(연봉데이터리스트);
        excelUtilsBase.create(List.of(샘플데이터, 비교한_연봉리스트));

    }

    /**
     * 연봉 입력을 위한 엑셀 템플릿 업로드 API
     */
    @ApiOperation(value = "연봉 입력 엑셀 템플릿 업로드")
    @PostMapping("/excel-upload.do")
    public ModelAndView uploadSalaryTemplate(@RequestPart("excelFile") MultipartFile excelFile, HttpServletRequest httpServletRequest) throws Exception {

        ExcelUtilsBase excelUtilsBase = ExcelUtilsFactory.getInstance(excelFile.getInputStream());
        List<SalaryEntity> uploadedSalaryList = excelUtilsBase.read(SalaryEntity.class);
        ModelAndView modelAndView = new ModelAndView("jsonView");
        modelAndView.addObject("result", salaryService.saveExcelDataToDB(uploadedSalaryList));
        return modelAndView;

    }

    @PutMapping
    public ResponseEntity<CommonResponse.ApiResult<List<SalaryVO>>> bulkUpdate(@RequestBody Map<String, SalaryVO> salaryMaps) throws Exception {
        List<SalaryVO> salaryVOList = salaryMaps.values().stream().collect(Collectors.toList());
        List<SalaryEntity> salaryEntityList = salaryVOList.stream()
                .map(dto -> modelMapper.map(dto, SalaryEntity.class))
                .collect(Collectors.toList());
        salaryService.updateSalary(salaryEntityList);
        Map<String, SalaryEntity> fetchSalary = salaryService.getAllSalariesMap();
        List<SalaryVO> response = fetchSalary.values().stream()
                .map(entity -> modelMapper.map(entity, SalaryVO.class))
                .collect(Collectors.toList());
        return ResponseEntity.ok(CommonResponse.success(response));
    }

    @GetMapping("/transfer-salary-info")
    public List<SalaryVO> transferSalaryInfo() throws Exception {
        return salaryService.getAllSalaries();
    }
}
