/*
 * @author Dongmin.lee
 * @since 2023-03-20
 * @version 23.03.20
 * @see <pre>
 *  Copyright (C) 2007 by 313 DEV GRP, Inc - All Rights Reserved
 *  Unauthorized copying of this file, via any medium is strictly prohibited
 *  Proprietary and confidential
 *  Written by 313 developer group <313@313.co.kr>, December 2010
 * </pre>
 */
package com.arms.api.util.dynamicscheduler.service;

import com.arms.api.globaltreemap.model.GlobalTreeMapEntity;
import com.arms.api.globaltreemap.service.GlobalTreeMapService;
import com.arms.api.jira.jiraproject.service.JiraProject;
import com.arms.api.jira.jiraproject_pure.model.JiraProjectPureEntity;
import com.arms.api.jira.jiraproject_pure.service.JiraProjectPure;
import com.arms.api.jira.jiraserver.model.JiraServerEntity;
import com.arms.api.jira.jiraserver.service.JiraServer;
import com.arms.api.jira.jiraserver_pure.model.JiraServerPureEntity;
import com.arms.api.jira.jiraserver_pure.service.JiraServerPure;
import com.arms.api.product_service.pdservice_pure.model.PdServicePureEntity;
import com.arms.api.product_service.pdservice_pure.service.PdServicePure;
import com.arms.api.product_service.pdserviceversion.model.PdServiceVersionEntity;
import com.arms.api.product_service.pdserviceversion.service.PdServiceVersion;
import com.arms.api.requirement.reqstate.service.ReqState;
import com.arms.api.requirement.reqstate_category.service.ReqStateCategory;
import com.arms.api.requirement.reqstatus.model.CRUDType;
import com.arms.api.requirement.reqstatus.model.ReqStatusDTO;
import com.arms.api.requirement.reqstatus.model.ReqStatusEntity;
import com.arms.api.requirement.reqstatus.service.ReqStatus;
import com.arms.api.util.TreeServiceUtils;
import com.arms.api.util.UUIDUtil;
import com.arms.api.util.aspect.DwrSendAlarm;
import com.arms.api.util.aspect.MailSendAlarm;
import com.arms.api.util.aspect.SlackSendAlarm;
import com.arms.api.util.communicate.external.EngineService;
import com.arms.api.util.communicate.external.request.증분이슈수집RequestDTO;
import com.arms.api.util.communicate.external.response.jira.AlmIssue;
import com.arms.api.util.communicate.external.response.jira.CReqProperty;
import com.arms.api.util.communicate.internal.InternalService;
import com.arms.api.util.dynamicscheduler.model.SchedulerType;
import com.arms.egovframework.javaservice.treeframework.remote.Chat;
import com.arms.egovframework.javaservice.treeframework.service.TreeServiceImpl;
import com.arms.egovframework.javaservice.treeframework.util.StringUtils;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.modelmapper.ModelMapper;
import org.springframework.http.ResponseEntity;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;

import java.time.Duration;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.*;
import java.util.stream.Collectors;

import static com.arms.egovframework.javaservice.treeframework.TreeConstant.REQSTATUS_PREFIX_TABLENAME;


@RequiredArgsConstructor
@Service("스케쥴러")
@Slf4j
public class SchedulerServiceImpl extends TreeServiceImpl implements SchedulerService {

    private final PdServicePure pdServicePure;

    private final ReqStatus reqStatus;

    private final JiraServerPure jiraServerPure;

    private final JiraServer jiraServer;

    private final EngineService engineService;

    private final InternalService internalService;

    private final JiraProject jiraProject;

    private final JiraProjectPure jiraProjectPure;

    private final ReqState reqState;

    private final ReqStateCategory reqStateCategory;

    private final DateTimeFormatter timeFormatter = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");

    protected final Chat chat;

    protected final ModelMapper modelMapper;

    private final GlobalTreeMapService globalTreeMapService;

    private final PdServiceVersion pdServiceVersion;

    @Override
    @Async
    @DwrSendAlarm(messageOnStart = "전체 이슈 수집 요청")
    @SlackSendAlarm(messageOnStart = "전체 이슈 수집 요청을 하였습니다.")
    public void discoverByFullDateRange(SchedulerType schedulerEnum) throws Exception {
        String logMessagePrefix = "[ SchedulerServiceImpl :: executeSequentialSchedules ]";
        String shortUUID = UUIDUtil.createShortUUID();

        LocalDateTime startTime = LocalDateTime.now();
        log.info("[{}] :: {} :: {} :: started at {}", shortUUID, logMessagePrefix, schedulerEnum.getType(), startTime.format(timeFormatter));

        String yesterday = LocalDate.now().minusDays(1)
                .format(DateTimeFormatter.ofPattern("yyyy-MM-dd"));
        incrementSchedulesWithRange(schedulerEnum,null,yesterday);

        LocalDateTime endTime = LocalDateTime.now();
        Duration elapsed = Duration.between(startTime, endTime);
        log.info("[{}] :: {} :: {} :: finished at {} (completed in {} ms)", shortUUID, logMessagePrefix, schedulerEnum.getType(), endTime.format(timeFormatter), elapsed.toMillis());

    }

    @Override
    @Async
    @DwrSendAlarm(messageOnStart = "작일 이슈 수집 요청")
    @SlackSendAlarm(messageOnStart = "작일 이슈 수집 요청을 하였습니다.")
    public void discoverByPreviousDay(SchedulerType schedulerEnum) throws Exception {
        String yesterday = LocalDate.now().minusDays(1)
                .format(DateTimeFormatter.ofPattern("yyyy-MM-dd"));
        String logMessagePrefix = "[ SchedulerServiceImpl :: discoverByPreviousDay ]";
        String shortUUID = UUIDUtil.createShortUUID();

        LocalDateTime startTime = LocalDateTime.now();
        log.info("[{}] :: {} :: {} :: started at {}", shortUUID, logMessagePrefix, schedulerEnum.getType(), startTime.format(timeFormatter));

        this.incrementSchedulesWithRange(schedulerEnum, yesterday,  yesterday);

        LocalDateTime endTime = LocalDateTime.now();
        Duration elapsed = Duration.between(startTime, endTime);
        log.info("[{}] :: {} :: {} :: finished at {} (completed in {} ms)", shortUUID, logMessagePrefix, schedulerEnum.getType(), endTime.format(timeFormatter), elapsed.toMillis());


    }

    @Async
    @DwrSendAlarm(messageOnStart = "기간별 이슈 수집 요청")
    @SlackSendAlarm(messageOnStart = "기간별 이슈 수집 요청을 하였습니다.")
    public void discoverByDateRange(SchedulerType schedulerEnum, String startDate, String endDate) throws Exception {
        String logMessagePrefix = "[ SchedulerServiceImpl :: discoverByDateRange ]";
        String shortUUID = UUIDUtil.createShortUUID();

        LocalDateTime startTime = LocalDateTime.now();
        log.info("[{}] :: {} :: {} :: started at {}", shortUUID, logMessagePrefix, schedulerEnum.getType(), startTime.format(timeFormatter));

        this.incrementSchedulesWithRange(schedulerEnum, startDate,  endDate);

        LocalDateTime endTime = LocalDateTime.now();
        Duration elapsed = Duration.between(startTime, endTime);
        log.info("[{}] :: {} :: {} :: finished at {} (completed in {} ms)", shortUUID, logMessagePrefix, schedulerEnum.getType(), endTime.format(timeFormatter), elapsed.toMillis());

    }

    private List<PdServicePureEntity> reqStatus_엔진에_동기화_그리고_pdService목록리턴(String logMessagePrefix) throws Exception {
        List<PdServicePureEntity> pdServicePureEntities = pdServicePure.getNodesWithoutRoot(new PdServicePureEntity());

        for (PdServicePureEntity pdServicePureEntity : pdServicePureEntities) {
            Long pdServiceEntityCId = pdServicePureEntity.getC_id();

            List<ReqStatusEntity> searchReqStatusEntities = internalService.제품별_요구사항_상황_조회(REQSTATUS_PREFIX_TABLENAME + pdServiceEntityCId, new ReqStatusDTO());

            if (searchReqStatusEntities == null || searchReqStatusEntities.isEmpty()) {
                log.info("{} :: {} 제품의 요구사항이 존재하지 않습니다.", logMessagePrefix, pdServicePureEntity.getC_title());
                continue;
            }

            for (ReqStatusEntity reqStatusEntity : searchReqStatusEntities) {

                if(CRUDType.소프트_삭제.getType().equals(reqStatusEntity.getC_etc())){
                    continue;
                }

                Optional<JiraServerPureEntity> optionalJiraServerPureEntity = TreeServiceUtils.getNodeOptional(jiraServerPure, reqStatusEntity.getC_jira_server_link(), JiraServerPureEntity.class);

                optionalJiraServerPureEntity.ifPresentOrElse(jiraServerPureEntity -> {
                        if (isValidReqStatus(reqStatusEntity, jiraServerPureEntity, logMessagePrefix)) {

                            engineService.saveRequirementIssueKeyOnlyFirst(
                                    jiraServerPureEntity.getC_jira_server_etc(),
                                    reqStatusEntity.getC_title(),
                                    reqStatusEntity.getC_id(),
                                    reqStatusEntity.getC_issue_key(),
                                    reqStatusEntity.getC_pdservice_link(),
                                    convertVersionToArray(reqStatusEntity.getC_req_pdservice_versionset_link()),
                                    reqStatusEntity.getC_req_link(),
                                    reqStatusEntity.getC_jira_project_key(),
                                    CReqProperty.of(reqStatusEntity)
                            );
                        }
                    }
                    , () -> log.info("{} :: 지라서버가 조회되지 않습니다. C_ID = {}", logMessagePrefix, reqStatusEntity.getC_jira_server_link())
                );
            }
        }
        return pdServicePureEntities;
    }

    private void incrementSchedulesWithRange(SchedulerType schedulerEnum, String startDate, String endDate) throws Exception {

        String logMessagePrefix = "[ SchedulerServiceImpl :: saveReqStatusToElasticsearch ]";
        String shortUUID = UUIDUtil.createShortUUID();

        LocalDateTime startTime = LocalDateTime.now();
        log.info("[{}] :: {} :: {} :: started at {}", shortUUID, logMessagePrefix, schedulerEnum.getType(), startTime.format(timeFormatter));

        List<JiraServerPureEntity> nodesWithoutRoot = jiraServerPure.getNodesWithoutRoot(new JiraServerPureEntity());

        List<String> serverIds = nodesWithoutRoot.stream().map(JiraServerPureEntity::getC_jira_server_etc).collect(Collectors.toList());

        Map<String, List<String>> jiraProjectsByJiraServerIds = jiraProjectsByJiraServerId();

        reqStatus_엔진에_동기화_그리고_pdService목록리턴(logMessagePrefix);

        List<증분이슈수집RequestDTO.ServerProjectInfo> serverProjectInfos = serverIds.stream()
                .filter(serverId -> jiraProjectsByJiraServerIds.get(serverId) != null)
                .flatMap(serverId -> jiraProjectsByJiraServerIds.get(serverId).stream()
                        .map(projectKey -> 증분이슈수집RequestDTO.ServerProjectInfo.builder()
                                .serverId(serverId)
                                .projectKey(projectKey)
                                .build()))
                .collect(Collectors.toList());

        증분이슈수집RequestDTO requestDTO = 증분이슈수집RequestDTO.builder()
                .serverProjectInfos(serverProjectInfos)
                .startDate(startDate)
                .endDate(endDate)
                .build();

        engineService.증분이슈수집(requestDTO);

    }

    private Map<String,List<String>> jiraProjectsByJiraServerId() throws Exception{

        List<PdServiceVersionEntity> nodesWithoutRoot = pdServiceVersion.getNodesWithoutRoot(new PdServiceVersionEntity());

        List<Long> collect = nodesWithoutRoot.stream()
                .map(PdServiceVersionEntity::getC_id)
                .collect(Collectors.toList());

        List<Long> jiraProjectLinkByPdServiceVersionLink = globalTreeMapService.findAllByIds(collect, "pdserviceversion_link")
                .stream()
                .map(GlobalTreeMapEntity::getJiraproject_link)
                .filter(Objects::nonNull)
                .distinct().collect(Collectors.toList());

        return globalTreeMapService.findAllByIds(jiraProjectLinkByPdServiceVersionLink, "jiraproject_link")
                .stream().filter(a -> a.getJiraserver_link() != null).collect(Collectors.groupingBy(a -> {
                    JiraServerPureEntity jiraServerPureEntity = new JiraServerPureEntity();
                    jiraServerPureEntity.setC_id(a.getJiraserver_link());
                    try {
                        return jiraServerPure.getNode(jiraServerPureEntity).getC_jira_server_etc();
                    } catch (Exception e) {
                        throw new RuntimeException(e);
                    }
                }, Collectors.mapping(a -> {
                    JiraProjectPureEntity jiraProjectPureEntity = new JiraProjectPureEntity();
                    jiraProjectPureEntity.setC_id(a.getJiraproject_link());
                    try {
                        return jiraProjectPure.getNode(jiraProjectPureEntity).getC_jira_key();
                    } catch (Exception e) {
                        throw new RuntimeException(e);
                    }
                }, Collectors.toList())));

    }

    @Async
    @Override
    @DwrSendAlarm(messageOnStart = "각 제품서비스 별 요구사항 Status 업데이트")
    @SlackSendAlarm(messageOnStart = "각 제품서비스 별 요구사항 Status 업데이트가 시작 되었습니다.", messageOnEnd = "각 제품서비스 별 요구사항 Status 업데이트가 완료 되었습니다.")
    @MailSendAlarm(subject = "각 제품서비스 별 요구사항 Status 업데이트 완료.",content = "각 제품서비스 별 요구사항 Status 업데이트 완료.")
    public void updateReqStatusFromElasticsearch(SchedulerType schedulerEnum) throws Exception {
        String logMessagePrefix = "[ SchedulerServiceImpl :: updateReqStatusFromElasticsearch ]";
        String shortUUID = UUIDUtil.createShortUUID();

        LocalDateTime startTime = LocalDateTime.now();
        log.info("[{}] :: {} :: {} :: started at {}", shortUUID, logMessagePrefix, schedulerEnum.getType(), startTime.format(timeFormatter));

        List<PdServicePureEntity> pdServicePureEntities = pdServicePure.getNodesWithoutRoot(new PdServicePureEntity());

        for (PdServicePureEntity pdServicePureEntity : pdServicePureEntities) {
            Long pdServiceEntityCId = pdServicePureEntity.getC_id();

            List<ReqStatusEntity> searchReqStatusEntities = internalService.제품별_요구사항_상황_조회(REQSTATUS_PREFIX_TABLENAME + pdServiceEntityCId, new ReqStatusDTO());

            if (searchReqStatusEntities == null || searchReqStatusEntities.isEmpty()) {
                log.info("{} :: {} 제품의 요구사항이 존재하지 않습니다.", logMessagePrefix, pdServicePureEntity.getC_title());
                continue;
            }

            for (ReqStatusEntity reqStatusEntity : searchReqStatusEntities) {

                Optional<JiraServerPureEntity> optionalJiraServerPureEntity = TreeServiceUtils.getNodeOptional(jiraServerPure, reqStatusEntity.getC_jira_server_link(), JiraServerPureEntity.class);

                optionalJiraServerPureEntity.ifPresentOrElse(jiraServerPureEntity -> {

                    if (isValidReqStatus(reqStatusEntity, jiraServerPureEntity, logMessagePrefix)) {
                        AlmIssue jiraIssue = engineService.요구사항이슈_조회(Long.parseLong(jiraServerPureEntity.getC_jira_server_etc()), reqStatusEntity.getC_jira_project_key(), reqStatusEntity.getC_issue_key());

                        if (isValidJiraIssue(jiraIssue, reqStatusEntity, logMessagePrefix)) {
                            log.info("{} :: jiraIssue :: id = {} :: key = {} ", logMessagePrefix, jiraIssue.getRecentId(), jiraIssue.getKey());

                            updateReqStatusFromJiraIssue(jiraIssue, reqStatusEntity, logMessagePrefix);

                            ReqStatusDTO statusDTO = modelMapper.map(reqStatusEntity, ReqStatusDTO.class);

                            ResponseEntity<?> reqStatusUpdateResponse = internalService.요구사항_상황_수정하기(REQSTATUS_PREFIX_TABLENAME + pdServiceEntityCId, statusDTO);

                            if (reqStatusUpdateResponse.getStatusCode().is2xxSuccessful()) {
                                String message = " [ " + statusDTO.getC_jira_server_name() + " :: " + statusDTO.getC_jira_project_name() + " ] :: " + statusDTO.getC_issue_key() + " ALM 요구사항 이슈의 데이터와 동기화되었습니다.";
                                chat.sendMessageByEngine(message);
                            } else {
                                log.error(REQSTATUS_PREFIX_TABLENAME + pdServiceEntityCId + " :: 수정 오류 :: " + statusDTO.toString());
                            }

                        }
                    }
                }, () -> log.info("{} :: 지라서버가 조회되지 않습니다. C_ID = {}", logMessagePrefix, reqStatusEntity.getC_jira_server_link()));
            }
        }

        LocalDateTime endTime = LocalDateTime.now();
        Duration elapsed = Duration.between(startTime, endTime);
        log.info("[{}] :: {} :: {} :: finished at {} (completed in {} ms)", shortUUID, logMessagePrefix, schedulerEnum.getType(), endTime.format(timeFormatter), elapsed.toMillis());

    }

    @Async
    @Override
    @DwrSendAlarm(messageOnStart = "이슈생성 실패 항목 재시도")
    @SlackSendAlarm(messageOnStart = "이슈생성 실패 항목 재생성이 시작 되었습니다.", messageOnEnd = "이슈생성 실패 항목 재생성이 완료 되었습니다.")
    @MailSendAlarm(subject = "이슈생성 실패 항목 재생성",content = "이슈생성 실패 항목 재생성 완료")
    public void retryFailedReqStatusCreationToElasticsearch(SchedulerType schedulerEnum) throws Exception {

        String logMessagePrefix = "[ SchedulerServiceImpl :: retryFailedReqStatusCreationToElasticsearch ]";
        String shortUUID = UUIDUtil.createShortUUID();

        LocalDateTime startTime = LocalDateTime.now();
        log.info("[{}] :: {} :: {} :: started at {}", shortUUID, logMessagePrefix, schedulerEnum.getType(), startTime.format(timeFormatter));

        List<PdServicePureEntity> pdServicePureEntities = pdServicePure.getNodesWithoutRoot(new PdServicePureEntity());

        for (PdServicePureEntity pdServicePureEntity : pdServicePureEntities) {
            Long pdServiceEntityCId = pdServicePureEntity.getC_id();

            List<ReqStatusEntity> searchReqStatusEntities = internalService.제품별_요구사항_상황_조회(REQSTATUS_PREFIX_TABLENAME + pdServiceEntityCId, new ReqStatusDTO());

            if (searchReqStatusEntities == null || searchReqStatusEntities.isEmpty()) {
                log.info("{} :: {} 제품의 요구사항이 존재하지 않습니다.", logMessagePrefix, pdServicePureEntity.getC_title());
                continue;
            }

            List<ReqStatusEntity> filteredIssues = Optional.ofNullable(searchReqStatusEntities)
                    .orElse(Collections.emptyList())
                    .stream()
                    .filter(요구사항_이슈 -> 요구사항_이슈.getC_issue_delete_date() == null)
                    .filter(요구사항_이슈 -> 요구사항_이슈.getC_etc() != null && !StringUtils.equals(CRUDType.완료.getType(), 요구사항_이슈.getC_etc()))
                    .collect(Collectors.toList());

            filteredIssues.forEach(요구사항_이슈 -> reqStatus.ALM서버_요구사항_처리_및_REQSTATUS_업데이트(요구사항_이슈, pdServiceEntityCId));
        }

        LocalDateTime endTime = LocalDateTime.now();
        Duration elapsed = Duration.between(startTime, endTime);
        log.info("[{}] :: {} :: {} :: finished at {} (completed in {} ms)", shortUUID, logMessagePrefix, schedulerEnum.getType(), endTime.format(timeFormatter), elapsed.toMillis());
    }

    @Async
    @Override
    @DwrSendAlarm(messageOnStart = "StatusMapping 데이터 캐싱")
    @SlackSendAlarm(messageOnStart = "StatusMapping 데이터 캐싱이 시작 되었습니다.", messageOnEnd = "StatusMapping 데이터 캐싱이 완료 되었습니다.")
    @MailSendAlarm(subject = "StatusMapping 데이터 캐싱",content = "StatusMapping 데이터 캐싱 완료")
    public String cacheStatusMappingData() throws Exception {
        String logMessagePrefix = "[ SchedulerServiceImpl :: cacheStatusMappingData ]";
        String shortUUID = UUIDUtil.createShortUUID();

        LocalDateTime startTime = LocalDateTime.now();
        log.info("[{}] :: {} :: started at {}", shortUUID, logMessagePrefix, startTime.format(timeFormatter));

        jiraServer.cacheStatusMappingData(new JiraServerEntity());
        reqState.cacheArmsStateData();
        reqStateCategory.cacheArmsStateCategoryData();

        LocalDateTime endTime = LocalDateTime.now();
        Duration elapsed = Duration.between(startTime, endTime);
        log.info("[{}] :: {} :: finished at {} (completed in {} ms)", shortUUID, logMessagePrefix, endTime.format(timeFormatter), elapsed.toMillis());
        return "success";
    }

    private boolean isValidReqStatus(ReqStatusEntity reqStatusEntity, JiraServerPureEntity jiraServerPureEntity, String logMessagePrefix) {

        if (isIssueKeyNull(reqStatusEntity, logMessagePrefix)) {
            return false;
        }

        if (isIssueDeleted(reqStatusEntity, logMessagePrefix)) {
            return false;
        }

        if (isVersionSetLinkMissing(reqStatusEntity, jiraServerPureEntity, logMessagePrefix)) {
            return false;
        }

        return true;
    }

    private boolean isIssueKeyNull(ReqStatusEntity reqStatusEntity, String logMessagePrefix) {
        if (reqStatusEntity.getC_issue_key() == null) {
            log.info("{} :: 해당 요구사항은 ALM 서버에 생성되지 않은 요구사항입니다. 확인이 필요합니다. C_ID = {}", logMessagePrefix, reqStatusEntity.getC_id());
            return true;
        }
        return false;
    }

    private boolean isIssueDeleted(ReqStatusEntity reqStatusEntity, String logMessagePrefix) {
        if (reqStatusEntity.getC_issue_delete_date() != null) {
            log.info("{} :: 해당 요구사항은 A-RMS 에서 삭제된 요구사항입니다. C_ID = {}", logMessagePrefix, reqStatusEntity.getC_id());
            return true;
        }
        return false;
    }

    private boolean isVersionSetLinkMissing(ReqStatusEntity reqStatusEntity, JiraServerPureEntity jiraServerPureEntity, String logMessagePrefix) {
        if (reqStatusEntity.getC_req_pdservice_versionset_link() == null || reqStatusEntity.getC_req_pdservice_versionset_link().isEmpty()) {
            log.info("{} :: 해당 요구사항은 버전 데이터를 가지고 있지 않습니다. C_ID = {}, name = {}, issue_key = {}", logMessagePrefix, reqStatusEntity.getC_id(), jiraServerPureEntity.getC_jira_server_name(), reqStatusEntity.getC_issue_key());
            return true;
        }
        return false;
    }

    private void updateReqStatusFromJiraIssue(AlmIssue jiraIssue, ReqStatusEntity reqStatusEntity, String logMessagePrefix) {
        log.info("{} :: jiraIssue :: id = {} :: key = {} ", logMessagePrefix, jiraIssue.getRecentId(), jiraIssue.getKey());

        if (jiraIssue.getAssignee() != null && jiraIssue.getAssignee().getDisplayName() != null) {
            log.info("{} :: jiraIssue 담당자 이름 = {}", logMessagePrefix, jiraIssue.getAssignee().getDisplayName());
            reqStatusEntity.setC_issue_assignee(jiraIssue.getAssignee().getDisplayName());
        }

        if (jiraIssue.getPriority() != null && jiraIssue.getPriority().getName() != null) {
            log.info("{} :: jiraIssue 우선순위 = {}", logMessagePrefix, jiraIssue.getPriority().getName());
            reqStatusEntity.setC_issue_priority_link(Long.valueOf(jiraIssue.getPriority().getId()));
            reqStatusEntity.setC_issue_priority_name(jiraIssue.getPriority().getName());
        }

        if (jiraIssue.getReporter() != null && jiraIssue.getReporter().getDisplayName() != null) {
            log.info("{} :: jiraIssue 보고자 = {}", logMessagePrefix, jiraIssue.getReporter().getDisplayName());
            reqStatusEntity.setC_issue_reporter(jiraIssue.getReporter().getDisplayName());
        }

        if (jiraIssue.getResolution() != null && jiraIssue.getResolution().getName() != null) {
            log.info("{} :: jiraIssue 해결책 = {}", logMessagePrefix, jiraIssue.getResolution().getName());
            reqStatusEntity.setC_issue_resolution_name(jiraIssue.getResolution().getName());
        }

        if (jiraIssue.getStatus() != null && jiraIssue.getStatus().getName() != null) {
            log.info("{} :: jiraIssue 상태 = {}", logMessagePrefix, jiraIssue.getStatus().getName());
            reqStatusEntity.setC_issue_status_link(Long.valueOf(jiraIssue.getStatus().getId()));
            reqStatusEntity.setC_issue_status_name(jiraIssue.getStatus().getName());
        }
    }

    private boolean isValidJiraIssue(AlmIssue jiraIssue, ReqStatusEntity reqStatusEntity, String logMessagePrefix) {

        if (isJiraIssueNull(jiraIssue, reqStatusEntity, logMessagePrefix)) {
            return false;
        }

        if (isJiraIssueKeyNull(jiraIssue, logMessagePrefix)) {
            return false;
        }

        if (isJiraIssueNotAvailable(jiraIssue, logMessagePrefix)) {
            return false;
        }

        return true;
    }

    private boolean isJiraIssueNull(AlmIssue jiraIssue, ReqStatusEntity reqStatusEntity, String logMessagePrefix) {
        if (jiraIssue == null) {
            log.info("{} :: 지라이슈가 조회되지 않습니다. 조회키 = {}", logMessagePrefix, reqStatusEntity.getC_issue_key());
            return true;
        }
        return false;
    }

    private boolean isJiraIssueKeyNull(AlmIssue jiraIssue, String logMessagePrefix) {
        if (jiraIssue.getKey() == null) {
            log.info("{} :: jiraIssue = null 이며, 확인이 필요합니다.", logMessagePrefix);
            return true;
        }
        return false;
    }

    private boolean isJiraIssueNotAvailable(AlmIssue jiraIssue, String logMessagePrefix) {
        AlmIssue.상태 status = jiraIssue.getStatus();
        if (status == null||StringUtils.equals(status.getName(), "해당 요구사항은 지라서버에서 조회가 되지 않는 상태입니다.")) {
            log.info("{} :: 해당 요구사항은 지라서버에서 조회가 되지 않는 상태입니다. jiraIssue = {}", logMessagePrefix, jiraIssue.getKey());
            return true;
        }
        return false;
    }



    private Long[] convertVersionToArray(String versionList) {
        return Arrays.stream(versionList.split("[\\[\\],\"]"))
                .filter(s -> !s.isEmpty())
                .map(Long::valueOf)
                .toArray(Long[]::new);
    }
}
