package com.arms.api.requirement.reqadd.kafka;

import com.arms.config.KafkaLagMonitor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.util.HashMap;
import java.util.Map;

/**
 * Kafka 모니터링 컨트롤러
 * 
 * Kafka 토픽의 Consumer Lag을 조회하는 API 제공
 * 
 * @author 313DEVGRP
 * @since 2025
 */
@RestController
@RequestMapping("/api/kafka")
public class KafkaMonitorController {

    private final Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private KafkaLagMonitor kafkaLagMonitor;

    /**
     * 특정 토픽의 전체 Lag 조회
     * 
     * GET /api/kafka/lag/{topic}
     * 
     * @param topic 토픽 이름
     * @return 토픽의 전체 Lag 정보
     */
    @GetMapping("/lag/{topic}")
    public ResponseEntity<Map<String, Object>> getTopicLag(@PathVariable String topic) {
        logger.info("Getting lag for topic: {}", topic);
        
        long totalLag = kafkaLagMonitor.getConsumerLag(topic);
        
        Map<String, Object> response = new HashMap<>();
        response.put("topic", topic);
        response.put("totalLag", totalLag);
        response.put("timestamp", System.currentTimeMillis());
        
        return ResponseEntity.ok(response);
    }

    /**
     * 특정 토픽의 파티션별 상세 Lag 조회
     * 
     * GET /api/kafka/lag/{topic}/detail
     * 
     * @param topic 토픽 이름
     * @return 파티션별 Lag 상세 정보
     */
    @GetMapping("/lag/{topic}/detail")
    public ResponseEntity<Map<String, Object>> getDetailedTopicLag(@PathVariable String topic) {
        logger.info("Getting detailed lag for topic: {}", topic);
        
        Map<Integer, Long> detailedLag = kafkaLagMonitor.getDetailedLag(topic);
        long totalLag = detailedLag.values().stream().mapToLong(Long::longValue).sum();
        
        Map<String, Object> response = new HashMap<>();
        response.put("topic", topic);
        response.put("totalLag", totalLag);
        response.put("partitionCount", detailedLag.size());
        response.put("partitionLags", detailedLag);
        response.put("timestamp", System.currentTimeMillis());
        
        return ResponseEntity.ok(response);
    }

    /**
     * 특정 토픽의 전체 상태 조회
     * 
     * GET /api/kafka/status/{topic}
     * 
     * @param topic 토픽 이름
     * @return Consumer 상태 정보 (Lag, 파티션 수, 상태 등)
     */
    @GetMapping("/status/{topic}")
    public ResponseEntity<Map<String, Object>> getConsumerStatus(@PathVariable String topic) {
        logger.info("Getting consumer status for topic: {}", topic);
        
        Map<String, Object> status = kafkaLagMonitor.getConsumerStatus(topic);
        
        return ResponseEntity.ok(status);
    }

    /**
     * REQADD 토픽의 Lag 조회 (편의 메서드)
     * 
     * GET /api/kafka/lag/reqadd
     * 
     * @return REQADD 토픽의 Lag 정보
     */
    @GetMapping("/lag/reqadd")
    public ResponseEntity<Map<String, Object>> getReqAddLag() {
        return getTopicLag("REQADD");
    }

    /**
     * 헬스체크
     * 
     * GET /api/kafka/health
     * 
     * @return Kafka 모니터링 가능 여부
     */
    @GetMapping("/health")
    public ResponseEntity<Map<String, Object>> health() {
        Map<String, Object> response = new HashMap<>();
        response.put("status", "UP");
        response.put("service", "kafka-monitor");
        response.put("timestamp", System.currentTimeMillis());
        
        return ResponseEntity.ok(response);
    }
}
