package com.arms.api.analysis.time.service;

import com.arms.api.analysis.time.model.dto.TimeDTO;
import com.arms.api.analysis.time.model.RidgeLineVO;
import com.arms.api.analysis.time.model.vo.AlmIssueStatVO;
import com.arms.api.analysis.time.model.vo.AlmIssuesByVersionVO;
import com.arms.api.analysis.time.model.vo.HeatMapVO;
import com.arms.api.analysis.time.model.vo.ScatterChartVO;
import com.arms.api.util.communicate.external.AggregationService;
import com.arms.api.util.communicate.external.EngineService;
import com.arms.api.util.communicate.external.response.jira.AlmIssue;
import com.arms.api.util.communicate.internal.InternalService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

@Service
@RequiredArgsConstructor
@Slf4j
public class TimeServiceImpl implements TimeService {

    private final InternalService internalService;

    private final EngineService engineService;

    private final AggregationService aggregationService;


    @Override
    public HeatMapVO getHeatMapDataByUpdated(TimeDTO timeDTO) {
        return engineService.getHeatMapDataByUpdated(timeDTO).getBody();

    }

    @Override
    public List<ScatterChartVO> getScatterChartData(TimeDTO timeDTO) {
        return engineService.getScatterData(timeDTO).getBody();
    }

    @Override
    public List<AlmIssueStatVO> getMultiCombinationChartData(TimeDTO timeDTO) {
        return aggregationService.getMultiCombinationChartData(timeDTO).getBody();
    }

    @Override
    public List<AlmIssuesByVersionVO> getUpdatedPdServiceVersionAlmIssueByDate(TimeDTO timeDTO) {
        ResponseEntity<List<AlmIssue>> searchDateRangeData = aggregationService.getUpdatedIssueByDateRange(timeDTO);

        Map<Long, List<AlmIssue>> versionGroupedResult = Optional.ofNullable(searchDateRangeData.getBody())
                .orElseGet(Collections::emptyList)
                .stream()
                .flatMap(issue -> issue.flattenPerVersionByVersions().stream())
                .collect(Collectors.groupingBy(AlmIssue::getPdServiceVersion));

        return versionGroupedResult.entrySet()
                .stream()
                .map(entry -> AlmIssuesByVersionVO.builder()
                        .pdServiceVersion(entry.getKey())
                        .issues(entry.getValue())
                        .build())
                .collect(Collectors.toList());
    }

    @Override
    public List<RidgeLineVO> getRidgeLineData(TimeDTO timeDTO) {
        return aggregationService.getRidgeLineData(timeDTO).getBody();
    }

}
