/*
 * Decompiled with CFR 0.152.
 */
package com.arms.api.wiki.service;

import com.arms.api.wiki.dto.TextOperation;
import com.arms.api.wiki.util.OtUtils;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.locks.ReentrantLock;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.ListOperations;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.core.ValueOperations;
import org.springframework.data.redis.core.script.RedisScript;
import org.springframework.data.redis.serializer.SerializationException;
import org.springframework.stereotype.Service;

@Service
public class OtService {
    private static final Logger logger = Logger.getLogger(OtService.class.getName());
    private static final int MAX_HISTORY_SIZE_PER_DOC = 500;
    private static final String CLUSTER_KEY_FORMAT = "doc:{%s}:%s:%s";
    private final ReentrantLock serviceLock = new ReentrantLock();
    private final RedisTemplate<String, Object> redisTemplate;
    private final ValueOperations<String, Object> valueOperations;
    private final ListOperations<String, Object> historyListOperations;
    private final RedisScript<Boolean> updateContentAndHistoryScript;
    private final ObjectMapper objectMapper;

    @Autowired
    public OtService(RedisTemplate<String, Object> redisTemplate, RedisScript<Boolean> updateContentAndHistoryScript, ObjectMapper objectMapper) {
        this.redisTemplate = redisTemplate;
        this.valueOperations = redisTemplate.opsForValue();
        this.historyListOperations = redisTemplate.opsForList();
        this.updateContentAndHistoryScript = updateContentAndHistoryScript;
        this.objectMapper = objectMapper;
        logger.info("OtService initialized.");
    }

    private String getContentKey(String sessionId, String documentId) {
        return String.format(CLUSTER_KEY_FORMAT, sessionId, "content", documentId);
    }

    private String getHistoryKey(String sessionId, String documentId) {
        return String.format(CLUSTER_KEY_FORMAT, sessionId, "history", documentId);
    }

    public String getDocumentContent(String sessionId, String documentId) {
        String contentKey = this.getContentKey(sessionId, documentId);
        try {
            Object content = this.valueOperations.get((Object)contentKey);
            return content instanceof String ? (String)content : "";
        }
        catch (Exception e) {
            logger.log(Level.SEVERE, String.format("Redis error getting content for key [%s]: %s", contentKey, e.getMessage()), e);
            return "";
        }
    }

    public int getRevision(String sessionId, String documentId) {
        String historyKey = this.getHistoryKey(sessionId, documentId);
        try {
            Long size = this.historyListOperations.size((Object)historyKey);
            return size != null ? size.intValue() : 0;
        }
        catch (Exception e) {
            logger.log(Level.SEVERE, String.format("Redis error getting size for key [%s]: %s", historyKey, e.getMessage()), e);
            return 0;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public TextOperation receiveOperation(String sessionId, String documentId, int clientRevision, TextOperation operation) throws IllegalArgumentException {
        this.serviceLock.lock();
        String contentKey = this.getContentKey(sessionId, documentId);
        String historyKey = this.getHistoryKey(sessionId, documentId);
        try {
            String currentContent = this.getDocumentContent(sessionId, documentId);
            int serverRevision = this.getRevision(sessionId, documentId);
            logger.info(String.format("[Session: %s, Doc: %s] Received op based on client rev %d (Server rev: %d). Op: %s", sessionId, documentId, clientRevision, serverRevision, operation));
            if (clientRevision < 0 || clientRevision > serverRevision) {
                throw new IllegalArgumentException(String.format("[Session: %s, Doc: %s] Invalid client revision: %d. Server revision is: %d.", sessionId, documentId, clientRevision, serverRevision));
            }
            ArrayList<TextOperation> concurrentOps = new ArrayList<TextOperation>();
            if (clientRevision < serverRevision) {
                try {
                    List rawOps = this.historyListOperations.range((Object)historyKey, (long)clientRevision, (long)(serverRevision - 1));
                    if (rawOps != null) {
                        for (Object rawOp : rawOps) {
                            if (!(rawOp instanceof String)) {
                                logger.warning(String.format("[Session: %s, Doc: %s] Unexpected non-string type found in history: %s", sessionId, documentId, rawOp != null ? rawOp.getClass().getName() : "null"));
                                continue;
                            }
                            String opJson = (String)rawOp;
                            try {
                                List opsList = (List)this.objectMapper.readValue(opJson, (TypeReference)new /* Unavailable Anonymous Inner Class!! */);
                                concurrentOps.add(new TextOperation(opsList));
                            }
                            catch (JsonProcessingException e) {
                                logger.warning(String.format("[Session: %s, Doc: %s] Failed to parse operation JSON from history: %s. JSON: %s", sessionId, documentId, e.getMessage(), opJson));
                                throw new IllegalStateException("Invalid operation format found in Redis history list for key: " + historyKey, e);
                            }
                        }
                    }
                }
                catch (SerializationException e) {
                    logger.log(Level.SEVERE, String.format("[Session: %s, Doc: %s] Redis DESERIALIZATION error getting concurrent ops (rev %d to %d) for key [%s]: %s", sessionId, documentId, clientRevision, serverRevision - 1, historyKey, e.getMessage()), e);
                    throw new RuntimeException("Failed to deserialize concurrent operations from Redis history.", e);
                }
                catch (Exception e) {
                    logger.log(Level.SEVERE, String.format("[Session: %s, Doc: %s] Generic Redis error getting concurrent ops (rev %d to %d) for key [%s]: %s", sessionId, documentId, clientRevision, serverRevision - 1, historyKey, e.getMessage()), e);
                    throw new RuntimeException("Failed to retrieve concurrent operations from Redis history.", e);
                }
            }
            logger.fine(String.format("[Session: %s, Doc: %s] Found %d concurrent operations in Redis history to transform against.", sessionId, documentId, concurrentOps.size()));
            TextOperation transformedOp = operation;
            for (TextOperation concurrentOp : concurrentOps) {
                logger.fine(String.format("[Session: %s, Doc: %s] Transforming against concurrent op: %s", sessionId, documentId, concurrentOp));
                List result = OtUtils.transform((TextOperation)transformedOp, (TextOperation)concurrentOp);
                transformedOp = (TextOperation)result.get(0);
                logger.fine(String.format("[Session: %s, Doc: %s] Result after transform: %s", sessionId, documentId, transformedOp));
            }
            logger.info(String.format("[Session: %s, Doc: %s] Attempting to apply op [Rev %d]: %s to current doc content (length %d): '%s'", sessionId, documentId, serverRevision, transformedOp, currentContent.length(), currentContent));
            String newContent = OtUtils.apply((String)currentContent, (TextOperation)transformedOp);
            logger.info(String.format("[Session: %s, Doc: %s] Document content after applying transformed op: '%s'", sessionId, documentId, newContent));
            try {
                String transformedOpJson = this.objectMapper.writeValueAsString((Object)transformedOp.getOps());
                List<String> keys = List.of(contentKey, historyKey);
                this.redisTemplate.execute(this.updateContentAndHistoryScript, keys, new Object[]{newContent, transformedOpJson, String.valueOf(500)});
                logger.fine(String.format("[Session: %s, Doc: %s] Successfully updated content and added op JSON to history via Lua script. New revision: %d", sessionId, documentId, serverRevision + 1));
            }
            catch (JsonProcessingException e) {
                logger.log(Level.SEVERE, String.format("[Session: %s, Doc: %s] Failed to serialize transformed operation to JSON: %s", sessionId, documentId, transformedOp), e);
                throw new RuntimeException("Failed to serialize operation for Redis history.", e);
            }
            catch (Exception e) {
                logger.log(Level.SEVERE, String.format("[Session: %s, Doc: %s] Redis error executing Lua script for key [%s] and history [%s]: %s", sessionId, documentId, contentKey, historyKey, e.getMessage()), e);
                throw new RuntimeException("Failed to atomically update Redis content and history.", e);
            }
            TextOperation textOperation = transformedOp;
            return textOperation;
        }
        finally {
            this.serviceLock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void setDocumentContent(String sessionId, String documentId, String content) {
        this.serviceLock.lock();
        String contentKey = this.getContentKey(sessionId, documentId);
        String historyKey = this.getHistoryKey(sessionId, documentId);
        try {
            this.valueOperations.set((Object)contentKey, (Object)(content != null ? content : ""));
            this.redisTemplate.delete((Object)historyKey);
            logger.info(String.format("[Session: %s, Doc: %s] Document content set directly in Redis key [%s]. History list [%s] deleted. New revision: 0", sessionId, documentId, contentKey, historyKey));
        }
        catch (Exception e) {
            logger.log(Level.SEVERE, String.format("[Session: %s, Doc: %s] Redis error setting content [%s] or deleting history [%s]: %s", sessionId, documentId, contentKey, historyKey, e.getMessage()), e);
        }
        finally {
            this.serviceLock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void resetSessionDocument(String sessionId, String documentId) {
        this.serviceLock.lock();
        String contentKey = this.getContentKey(sessionId, documentId);
        String historyKey = this.getHistoryKey(sessionId, documentId);
        List<String> keysToDelete = List.of(contentKey, historyKey);
        try {
            Long deletedCount = this.redisTemplate.delete(keysToDelete);
            if (deletedCount != null && deletedCount > 0L) {
                logger.info(String.format("[Session: %s, Doc: %s] Document state reset in Redis. Deleted keys: %s", sessionId, documentId, keysToDelete));
            } else {
                logger.warning(String.format("[Session: %s, Doc: %s] Attempted to reset non-existent document state in Redis. Keys not found: %s", sessionId, documentId, keysToDelete));
            }
        }
        catch (Exception e) {
            logger.log(Level.SEVERE, String.format("[Session: %s, Doc: %s] Redis error deleting keys %s: %s", sessionId, documentId, keysToDelete, e.getMessage()), e);
        }
        finally {
            this.serviceLock.unlock();
        }
    }

    public List<TextOperation> getOperationHistory(String sessionId, String documentId) {
        String historyKey = this.getHistoryKey(sessionId, documentId);
        try {
            List rawOps = this.historyListOperations.range((Object)historyKey, 0L, -1L);
            if (rawOps != null) {
                ArrayList<TextOperation> history = new ArrayList<TextOperation>(rawOps.size());
                for (Object rawOp : rawOps) {
                    if (!(rawOp instanceof String)) {
                        logger.warning(String.format("[Session: %s, Doc: %s] Unexpected non-string type found in full history: %s", sessionId, documentId, rawOp != null ? rawOp.getClass().getName() : "null"));
                        continue;
                    }
                    String opJson = (String)rawOp;
                    try {
                        List opsList = (List)this.objectMapper.readValue(opJson, (TypeReference)new /* Unavailable Anonymous Inner Class!! */);
                        history.add(new TextOperation(opsList));
                    }
                    catch (JsonProcessingException e) {
                        logger.warning(String.format("[Session: %s, Doc: %s] Failed to parse operation JSON from full history: %s. JSON: %s", sessionId, documentId, e.getMessage(), opJson));
                    }
                }
                return history;
            }
            return Collections.emptyList();
        }
        catch (Exception e) {
            logger.log(Level.SEVERE, String.format("[Session: %s, Doc: %s] Redis error getting full history for key [%s]: %s", sessionId, documentId, historyKey, e.getMessage()), e);
            return Collections.emptyList();
        }
    }
}

