package com.arms.api.wiki.controller;

import com.arms.api.util.response.CommonResponse;
import com.arms.api.wiki.model.WikiVersionVO;
import com.arms.api.wiki.model.WikiDTO;
import com.arms.api.wiki.model.WikiVO;
import com.arms.api.wiki.model.WikiVersionsVO;
import com.arms.api.wiki.service.WikiService;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import java.util.List;

@RestController
@RequestMapping("/engine/wiki")
@Slf4j
@AllArgsConstructor
public class WikiController {

    private final WikiService wikiService;

    @GetMapping("/{wikiId}")
    public ResponseEntity<CommonResponse.ApiResult<WikiVO>> getWiki(@PathVariable("wikiId") String wikiId) {
        log.info("[ WikiController :: getWiki ] wikiId => {}", wikiId);
        return ResponseEntity.ok(CommonResponse.success(wikiService.getWiki(wikiId)));
    }

    @GetMapping("/{wikiId}/{version}")
    public ResponseEntity<CommonResponse.ApiResult<WikiVO>> getWiki(@PathVariable("wikiId") String wikiId, @PathVariable("version") Long version) {
        log.info("[ WikiController :: getWiki ] wikiId => {}, version => {}", wikiId, version);
        return ResponseEntity.ok(CommonResponse.success(wikiService.getWiki(wikiId, version)));
    }

    @GetMapping("/version/{wikiId}")
    public ResponseEntity<CommonResponse.ApiResult<WikiVersionsVO>> getWikiVersion(@PathVariable("wikiId") String wikiId, @RequestParam("size") int size, @RequestParam("page") int page) {
        log.info("[ WikiController :: getWikiVersion ] wikiId => {}", wikiId);
        return ResponseEntity.ok(CommonResponse.success(wikiService.getWikiVersions(wikiId, size, page)));
    }

    @PostMapping("/create")
    public ResponseEntity<String> createWiki(@RequestBody WikiDTO wikiDTO) {
        log.info("[ WikiController :: createWiki ] id => {}", wikiDTO.getId());
        return ResponseEntity.ok(wikiService.createWiki(wikiDTO));
    }

    @PutMapping("/update")
    public ResponseEntity<String> updateWiki(@RequestBody WikiDTO wikiDTO) {
        log.info("[ WikiController :: updateWiki ] id => {}", wikiDTO.getId());
        return ResponseEntity.ok(wikiService.updateWiki(wikiDTO));
    }

    @PutMapping("/change-recent")
    public ResponseEntity<CommonResponse.ApiResult<String>> changeRecent(@RequestBody WikiDTO wikiDTO) {
        log.info("[ WikiController :: changeRecent ] id => {}", wikiDTO.getId());
        wikiService.changeRecent(wikiDTO);
        return ResponseEntity.ok(CommonResponse.success(wikiDTO.getId()));
    }

    @DeleteMapping("/{id}")
    public ResponseEntity<CommonResponse.ApiResult<String>> deleteWiki(@PathVariable("id") String deleteId) {
        log.info("[ WikiController :: deleteWiki ] id => {}", deleteId);
        return ResponseEntity.ok(CommonResponse.success(wikiService.deleteWiki(deleteId)));
    }
}
