package com.arms.api.project.strategy;

import com.arms.api.project.dto.ProjectDTO;
import com.arms.api.project.vo.ProjectVO;
import com.arms.api.serverinfo.model.ServerInfo;
import com.arms.api.serverinfo.service.ServerInfoService;
import com.arms.api.util.alm.JiraApi;
import com.arms.api.util.alm.JiraUtil;
import com.arms.api.util.errors.ErrorLogUtil;
import lombok.AllArgsConstructor;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.stereotype.Component;
import org.springframework.web.reactive.function.client.WebClient;

import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;

@Component
@AllArgsConstructor
public class CloudJiraProjectStrategy implements ProjectStrategy {

    private final JiraUtil jiraUtil;

    private final JiraApi jiraApi;

    private final ServerInfoService serverInfoService;

    @Override
    public ProjectVO getProject(ServerInfo serverInfo, String projectKeyOrId) {
        String endpoint = jiraApi.getEndpoint().getProject() + "/" + projectKeyOrId;
        WebClient webClient = jiraUtil.createJiraCloudCommunicator(serverInfo.getUri(), serverInfo.getUserId(), serverInfoService.getDecryptPasswordOrToken(serverInfo));

        try {
            ProjectDTO projectDTO = jiraUtil.get(webClient, endpoint, ProjectDTO.class).block();
            return this.convertProjectVO(projectDTO);
        }
        catch (Exception e) {
            String errorMessage = ErrorLogUtil.exceptionLoggingAndReturn(e, this.getClass().getName(),
                    String.format("%s[%s] :: getProject Error 프로젝트[%s] ",
                            serverInfo.getType(), serverInfo.getUri(), projectKeyOrId));
            throw new IllegalArgumentException(errorMessage);
        }
    }

    @Override
    public List<ProjectVO> getProjectList(ServerInfo serverInfo) {
        String endpoint = jiraApi.getEndpoint().getProject();
        WebClient webClient = jiraUtil.createJiraCloudCommunicator(serverInfo.getUri(), serverInfo.getUserId(), serverInfoService.getDecryptPasswordOrToken(serverInfo));

        try {
            List<ProjectDTO> projectList = jiraUtil.get(webClient, endpoint,
                    new ParameterizedTypeReference<List<ProjectDTO>>() {}).block();

            return Optional.ofNullable(projectList)
                    .orElse(Collections.emptyList())
                    .stream()
                    .filter(Objects::nonNull)
                    .map(this::convertProjectVO)
                    .collect(Collectors.toList());
        }
        catch (Exception e) {
            String errorMessage = ErrorLogUtil.exceptionLoggingAndReturn(e, this.getClass().getName(),
                    String.format("%s[%s] :: getProjectList Error", serverInfo.getType(), serverInfo.getUri()));
            throw new IllegalArgumentException(errorMessage);
        }
    }

    private ProjectVO convertProjectVO(ProjectDTO projectDTO) {
        return ProjectVO.builder()
                .id(projectDTO.getId())
                .key(projectDTO.getKey())
                .name(projectDTO.getName())
                .self(projectDTO.getSelf()).build();
    }
}