package com.arms.api.issue.priority.service;

import com.arms.api.issue.priority.model.IssuePriorityDTO;
import com.arms.api.issue.priority.strategy.*;
import com.arms.api.serverinfo.model.enums.ServerType;
import com.arms.api.serverinfo.model.ServerInfo;
import com.arms.api.serverinfo.service.ServerInfoService;
import com.arms.api.util.errors.ErrorCode;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.List;

@Slf4j
@Service
@RequiredArgsConstructor
public class IssuePriorityService {

    private ServerInfoService ServerInfoService;
    private CloudJiraIssuePriorityStrategy cloudJiraIssuePriorityStrategy;
    private OnpremiseJiraIssuePriorityStrategy onpremiseJiraIssuePriorityStrategy;
    private OnpremiseRedmineIssuePriorityStrategy onpremiseRedmineIssuePriorityStrategy;

    @Autowired
    public IssuePriorityService(ServerInfoService ServerInfoService,
                                CloudJiraIssuePriorityStrategy cloudJiraIssuePriorityStrategy,
                                OnpremiseJiraIssuePriorityStrategy onpremiseJiraIssuePriorityStrategy,
                                OnpremiseRedmineIssuePriorityStrategy onpremiseRedmineIssuePriorityStrategy) {

        this.ServerInfoService = ServerInfoService;
        this.cloudJiraIssuePriorityStrategy = cloudJiraIssuePriorityStrategy;
        this.onpremiseJiraIssuePriorityStrategy = onpremiseJiraIssuePriorityStrategy;
        this.onpremiseRedmineIssuePriorityStrategy = onpremiseRedmineIssuePriorityStrategy;
    }

    private IssuePriorityStrategyFactory settingIssuePriorityStrategy(ServerInfo serverInfo) {

        if (serverInfo == null || serverInfo.getType().isEmpty()) {
            log.error("settingIssuePriorityStrategy Error: serverInfo_유형 {}" , ErrorCode.SERVER_TYPE_INFO_ERROR.getErrorMsg());
            throw new IllegalArgumentException("settingIssuePriorityStrategy Error: serverInfo_유형 " + ErrorCode.SERVER_TYPE_INFO_ERROR.getErrorMsg());
        }

        ServerType serverType = ServerType.typeValueOf(serverInfo.getType());

        IssuePriorityStrategy issuePriorityStrategy;
        if (serverType == ServerType.CLOUD) {
            issuePriorityStrategy = this.cloudJiraIssuePriorityStrategy;
        }
        else if (serverType == ServerType.ON_PREMISS) {
            issuePriorityStrategy = this.onpremiseJiraIssuePriorityStrategy;
        }
        else if (serverType == ServerType.REDMINE_ON_PREMISS) {
            issuePriorityStrategy = this.onpremiseRedmineIssuePriorityStrategy;
        }        
        else {
            throw new IllegalArgumentException("settingIssuePriorityStrategy Error: 허용하지 않는 serverInfo_유형입니다. :: "+ serverType+ " :: " + ErrorCode.SERVER_TYPE_INFO_ERROR.getErrorMsg());
        }

        IssuePriorityStrategyFactory issuePriorityStrategyFactory = new IssuePriorityStrategyFactory();
        issuePriorityStrategyFactory.registerIssuePriorityStrategy(issuePriorityStrategy);

        return issuePriorityStrategyFactory;
    }

    public List<IssuePriorityDTO> getIssuePriorityList(String connectId) {

        ServerInfo serverInfo = ServerInfoService.verifyServerInfo(connectId);
        IssuePriorityStrategyFactory issuePriorityStrategy = settingIssuePriorityStrategy(serverInfo);

        List<IssuePriorityDTO> issuePriorityList = issuePriorityStrategy.getIssuePriorityList(serverInfo);
        return issuePriorityList;
    }

}
