package com.arms.egovframework.javaservice.gcframework.parser;

import com.arms.egovframework.javaservice.gcframework.model.AbstractContentVO;

import java.nio.charset.StandardCharsets;
import java.util.Base64;

/**
 * 다양한 파일 형식(JSON, YAML 등)의 내용을 특정 AbstractConfigVo 타입으로 파싱(역직렬화)하거나,
 * 반대로 AbstractConfigVo를 파일 내용으로 직렬화하는 공통 로직을 정의하는 추상 클래스
 * doParse와 doSerialize와 같은 추상 메서드를 통해 각 파서별 고유 로직을 하위 클래스에 위임
 */
public abstract class AbstractContentParser<T extends AbstractContentVO>{

    public abstract String getHandledType(); // "yml, yaml", "json" 등

    /**
     * 파일 내용을 지정된 VO 타입으로 파싱
     * @param rawContent 원본 파일 내용(Base64 디코딩 전 문자열일 수 있음)
     * @param targetVoType 파싱할 VO의 Class 타입
     * @return 파싱된 VO
     */
    public T parse(String rawContent, Class<T> targetVoType) {
        // 공통적인 전처리 (예: Base64 디코딩, 만약 RepositoryProvider가 Raw Content를 넘겨준다면)
        String decodedContent = new String(Base64.getDecoder().decode(rawContent), StandardCharsets.UTF_8);
        return doParse(decodedContent, targetVoType);
    }

    /**
     * VO를 파일 내용 (문자열)으로 직렬화
     * @param targetVO 직렬화할 VO
     * @return 파일 내용 문자열 (Base64 인코딩 전)
     */
    public String serialize(T targetVO) {
        String serializedContent = doSerialize(targetVO);
        // 공통적인 후처리 (예: Base64 인코딩, 만약 RepositoryProvider 가 Raw Content 를 받지 않는다면)
        return Base64.getEncoder().encodeToString(serializedContent.getBytes(StandardCharsets.UTF_8));
    }

    protected abstract T doParse(String decodedContent, Class<T> targetVoType);
    protected abstract String doSerialize(T targetVO);

}
