package com.arms.api.util;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.NoArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import org.yaml.snakeyaml.Yaml;

import javax.annotation.PostConstruct;
import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.net.HttpURLConnection;
import java.net.URL;
import java.util.ArrayList;
import java.util.Base64;
import java.util.List;
import java.util.Map;

@NoArgsConstructor
@Slf4j
@Component
public class GiteaFileUtil {

    @Value("${gitea.replace-url}")
    private String replaceUrlValue;
    private static String replaceUrl;

    @PostConstruct
    public void init() {
        replaceUrl = replaceUrlValue;
    }

    @SuppressWarnings("java:S2647")
    public static List<String> getJsonFileDownloadUrlsInDirectory(String directoryUrl, String branch, String username, String password) throws Exception {
        List<String> jsonFileUrls = new ArrayList<>();

        log.info("[ GiteaUtils :: getJsonFileDownloadUrlsInDirectory ] :: directoryUrl => {}, branch => {}", directoryUrl, branch);

        // 디렉토리 조회 API 호출
        HttpURLConnection connection = (HttpURLConnection) new URL(directoryUrl + "?ref=" + branch).openConnection();
        String auth = username + ":" + password;
        String encodedAuth = Base64.getEncoder().encodeToString(auth.getBytes());
        connection.setRequestProperty("Authorization", "Basic " + encodedAuth);
        connection.setRequestProperty("Accept", "application/json");

        if (connection.getResponseCode() != HttpURLConnection.HTTP_OK) {
            throw new RuntimeException("Failed to fetch directory: " + connection.getResponseMessage());
        }

        // 응답 파싱
        try (BufferedReader reader = new BufferedReader(new InputStreamReader(connection.getInputStream()))) {
            ObjectMapper mapper = new ObjectMapper();
            List<Map<String, Object>> files = mapper.readValue(reader, new TypeReference<List<Map<String, Object>>>() {
            });

            for (Map<String, Object> file : files) {
                String type = (String) file.get("type");
                String downloadUrl = (String) file.get("download_url");
                String parsedUrl = rewriteGiteaUrl(downloadUrl);
                if ("file".equals(type) && (parsedUrl.endsWith(".json"))) {
                    jsonFileUrls.add(parsedUrl);
                }
            }
            return jsonFileUrls;
        }
    }

    @SuppressWarnings("java:S2647")
    public static List<String> getJsonFileUrlsInDirectory(String directoryUrl, String branch, String username, String password) throws Exception {
        List<String> jsonFileUrls = new ArrayList<>();

        log.info("[ GiteaUtils :: getJsonFileUrlsInDirectory ] :: directoryUrl => {}, branch => {}", directoryUrl, branch);

        // 디렉토리 조회 API 호출
        HttpURLConnection connection = (HttpURLConnection) new URL(directoryUrl + "?ref=" + branch).openConnection();
        String auth = username + ":" + password;
        String encodedAuth = Base64.getEncoder().encodeToString(auth.getBytes());
        connection.setRequestProperty("Authorization", "Basic " + encodedAuth);
        connection.setRequestProperty("Accept", "application/json");

        if (connection.getResponseCode() != HttpURLConnection.HTTP_OK) {
            throw new RuntimeException("Failed to fetch directory: " + connection.getResponseMessage());
        }

        // 응답 파싱
        try (BufferedReader reader = new BufferedReader(new InputStreamReader(connection.getInputStream()))) {
            ObjectMapper mapper = new ObjectMapper();
            List<Map<String, Object>> files = mapper.readValue(reader, new TypeReference<List<Map<String, Object>>>() {
            });

            for (Map<String, Object> file : files) {
                String type = (String) file.get("type");
                String fileUrl = (String) file.get("url");
                String parsedUrl = rewriteGiteaUrl(fileUrl);
                if ("file".equals(type) && (parsedUrl.endsWith(".json?ref="+branch))) {
                    jsonFileUrls.add(parsedUrl);
                }
            }
            return jsonFileUrls;
        }
    }

    @SuppressWarnings("java:S2647")
    public static List<String> getYamlFilesFromDirectory(
            String directoryUrl, String branch, String username, String password) throws Exception {
        List<String> yamlFiles = new ArrayList<>();

        log.info("[ GiteaFileUtil :: getYamlFilesFromDirectory ] :: directoryUrl => {}, branch => {}", directoryUrl, branch);

        // 디렉토리 조회 API 호출
        HttpURLConnection connection = (HttpURLConnection) new URL(directoryUrl + "?ref=" + branch).openConnection();
        String auth = username + ":" + password;
        String encodedAuth = Base64.getEncoder().encodeToString(auth.getBytes());
        connection.setRequestProperty("Authorization", "Basic " + encodedAuth);

        if (connection.getResponseCode() != HttpURLConnection.HTTP_OK) {
            throw new RuntimeException("Failed to fetch directory: " + connection.getResponseMessage());
        }

        // 응답 파싱
        try (BufferedReader reader = new BufferedReader(new InputStreamReader(connection.getInputStream()))) {
            Yaml yaml = new Yaml();
            List<Map<String, Object>> files = yaml.load(reader);

            for (Map<String, Object> fileInfo : files) {
                String type = (String) fileInfo.get("type");
                String downloadUrl = (String) fileInfo.get("download_url");
                String parsedDownloadUrl = rewriteGiteaUrl(downloadUrl);
                log.info("[ GiteaFileUtil :: getYamlFilesFromDirectory ] :: type => {}, parsedDownloadUrl => {}", type, parsedDownloadUrl);
                // YAML 파일만 선택
                if ("file".equals(type) && (parsedDownloadUrl.endsWith(".yml") || parsedDownloadUrl.endsWith(".yaml"))) {
                    yamlFiles.add(parsedDownloadUrl);
                }
            }
        }
        return yamlFiles;
    }

    public static String extractFileNameWithoutExtension(String fileUrl) {
        String fullFileName = fileUrl.substring(fileUrl.lastIndexOf('/') + 1);
        return fullFileName.substring(0, fullFileName.lastIndexOf('.'));
    }

    public static String rewriteGiteaUrl(String url) {
        // 정규표현식: 프로토콜(http 또는 https)부터 /gitea까지 매칭
        String regex = "(https?://)([^/]+)(/gitea)";
        // 기존 URL에서 정규표현식을 사용해 부분 문자열 치환
        return url.replaceAll(regex, replaceUrl);
    }
}
