package com.arms.api.report.service;

import com.arms.api.issue.almapi.model.entity.AlmIssueEntity;
import com.arms.api.report.model.AssigneeIssueCountVO;
import com.arms.api.report.model.FullDataRequestDTO;
import com.arms.api.report.model.FullDataResponseDTO;
import com.arms.api.report.model.작업자_정보;
import com.arms.api.serverinfo.service.ServerInfoService;
import com.arms.egovframework.javaservice.esframework.esquery.filter.RangeQueryFilter;
import com.arms.egovframework.javaservice.esframework.model.dto.esquery.SearchDocDTO;
import com.arms.egovframework.javaservice.esframework.model.dto.request.SearchRequestDTO;
import com.arms.egovframework.javaservice.esframework.model.dto.request.AggregationRequestDTO;
import com.arms.egovframework.javaservice.esframework.model.dto.esquery.SubGroupFieldDTO;
import com.arms.egovframework.javaservice.esframework.model.vo.DocumentAggregations;
import com.arms.egovframework.javaservice.esframework.model.vo.DocumentBucket;
import com.arms.egovframework.javaservice.esframework.model.vo.DocumentResultWrapper;
import com.arms.egovframework.javaservice.esframework.repository.common.EsCommonRepositoryWrapper;
import com.arms.egovframework.javaservice.esframework.esquery.SimpleQuery;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import org.springframework.stereotype.Service;

import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

@Slf4j
@Service("리포트_서비스")
@AllArgsConstructor
public class ReportServiceImpl implements ReportService {

    private final EsCommonRepositoryWrapper<AlmIssueEntity> esCommonRepositoryWrapper;

    private ServerInfoService serverInfoService;

    @Override
    public List<작업자_정보> 작업자_목록_가져오기() {

        Map<String, String> serverIdToTypeMap = serverInfoService.getServerIdToTypeMap();

        DocumentAggregations documentAggregations = esCommonRepositoryWrapper.aggregateRecentDocs(
                SimpleQuery.aggregation(
                    AggregationRequestDTO.builder()
                        .mainField("assignee.assignee_emailAddress.keyword")
                        .mainFieldAlias("assignee")
                        .addGroup(
                            SubGroupFieldDTO.builder()
                                    .subFieldAlias("jiraServerId")
                                    .subField("jira_server_id")
                                    .build()
                            ,
                            SubGroupFieldDTO.builder()
                                    .subFieldAlias("displayName")
                                    .subField("assignee.assignee_displayName.keyword")
                                    .build()
                            ,
                            SubGroupFieldDTO.builder()
                                    .subFieldAlias("accountId")
                                    .subField("assignee.assignee_accountId.keyword")
                                    .build()
                        )
                    .build()
                ).andExistsQueryFilter("assignee")
        );

        List<DocumentBucket> deepestList = documentAggregations.deepestList();

        Set<작업자_정보> 작업자_목록 = new HashSet<>();

        if (documentAggregations.getTotalHits() == 0) {
            log.info("[ 리포트_서비스_프로세스 :: 작업자_목록_가져오기 ] :: 작업자 정보가 없습니다. 사이즈 => 0");
        }else{

            for (DocumentBucket documentBucket : deepestList) {

                작업자_정보 작업자 = new 작업자_정보();

                작업자.setEmailAddress(documentBucket.valueByName("emailAddress"));

                작업자.setServerType(serverIdToTypeMap.get(documentBucket.valueByName("jiraServerId")));

                작업자.setDisplayName(documentBucket.valueByName("displayName"));

                작업자.setAccountId(documentBucket.valueByName("accountId"));

                작업자_목록.add(작업자);
            }

        }

        if(작업자_목록.isEmpty()) {
            return new ArrayList<>(Collections.emptyList());
        } else {
            return new ArrayList<>(작업자_목록);
        }
    }

    private SimpleQuery<SearchDocDTO> 빌더에_넣을_공통_쿼리_목록_세팅(SimpleQuery<SearchDocDTO> simpleQuery,FullDataRequestDTO fullDataRequestDTO) {

        if (fullDataRequestDTO.getPdServiceId() != null) {
            simpleQuery.andTermQueryFilter("pdServiceId", fullDataRequestDTO.getPdServiceId());
        }

        if (fullDataRequestDTO.getPdServiceVersionIds() != null && !fullDataRequestDTO.getPdServiceVersionIds().isEmpty()) {
            simpleQuery.andTermsQueryFilter("pdServiceVersions", fullDataRequestDTO.getPdServiceVersionIds());
        }

        if (fullDataRequestDTO.getAlmProjectUrls() != null && !fullDataRequestDTO.getAlmProjectUrls().isEmpty()) {
            simpleQuery.andTermsQueryFilter("project.project_self.keyword", fullDataRequestDTO.getAlmProjectUrls());
        }

        // 작업자
        if (fullDataRequestDTO.getEmailAddress() != null && !fullDataRequestDTO.getEmailAddress().isEmpty()) {
            simpleQuery.andTermsQueryFilter("assignee.assignee_emailAddress.keyword", fullDataRequestDTO.getEmailAddress());
        }

        if (fullDataRequestDTO.getStartDate() != null && fullDataRequestDTO.getEndDate() == null) {
            simpleQuery.andRangeQueryFilter(RangeQueryFilter.of("updated").gte(fullDataRequestDTO.getStartDate()));
        }
        else if (fullDataRequestDTO.getStartDate() == null && fullDataRequestDTO.getEndDate() != null) {
            simpleQuery.andRangeQueryFilter(RangeQueryFilter.of("updated").lte(fullDataRequestDTO.getStartDate()));
        }
        else if (fullDataRequestDTO.getStartDate() != null && fullDataRequestDTO.getEndDate() != null) {
            simpleQuery.andRangeQueryFilter(RangeQueryFilter.of("updated").betweenDate(fullDataRequestDTO.getStartDate(),fullDataRequestDTO.getEndDate()));
        }
        return simpleQuery;
    }

    @Override
    public FullDataResponseDTO 이슈_목록_가져오기_스크롤API(FullDataRequestDTO fullDataRequestDTO) {

        SearchRequestDTO searchRequestDTO = new SearchRequestDTO();
        searchRequestDTO.setPage(fullDataRequestDTO.getPage());
        searchRequestDTO.setSize(fullDataRequestDTO.getSize());
        List<AlmIssueEntity> 스크롤API_활용_이슈_목록 
                = esCommonRepositoryWrapper.findRecentHits(
                        SimpleQuery.search(searchRequestDTO)
                            .andTermQueryFilter("pdServiceId", fullDataRequestDTO.getPdServiceId())
                            .andTermsQueryFilter("pdServiceVersions", fullDataRequestDTO.getPdServiceVersionIds())
                            .andTermsQueryFilter("project.project_self.keyword", fullDataRequestDTO.getAlmProjectUrls())
                            .andTermsQueryFilter("assignee.assignee_emailAddress.keyword", fullDataRequestDTO.getEmailAddress())
                            .andRangeQueryFilter(RangeQueryFilter.of("updated").betweenDate(fullDataRequestDTO.getStartDate(),fullDataRequestDTO.getEndDate()))
                ).toDocs(); // 스크롤API 필요

        FullDataResponseDTO 결과 = new FullDataResponseDTO();
        결과.setTotalHits((long) 스크롤API_활용_이슈_목록.size());
        결과.setIssueEntityList(스크롤API_활용_이슈_목록);
        return 결과;
    }

    // 실사용
    @Override
    public FullDataResponseDTO 이슈_목록_가져오기(FullDataRequestDTO fullDataRequestDTO) {

        List<AlmIssueEntity> 모든_이슈_목록 = new ArrayList<>();
        // upperKey 필드가 없는 지라이슈_엔티티를 담을 리스트
        List<AlmIssueEntity> ALM_생성_연결이슈 = new ArrayList<>();
        // upperKey 필드가 있는 지라이슈_엔티티를 담을 리스트
        List<AlmIssueEntity> 하위이슈_목록 = new ArrayList<>();
        List<AlmIssueEntity> 하위이슈만_존재하는_이슈의_요구사항이슈_목록 = new ArrayList<>();
        Set<String> 하위이슈_연결이슈_docIdSet = new HashSet<>();
        // 요구사항 이슈의 연결이슈 docIdSet
        Set<String> linkedIssuesDocIdSet = new HashSet<>();

        SearchRequestDTO searchRequestDTO = new SearchRequestDTO();
        searchRequestDTO.setPage(fullDataRequestDTO.getPage());
        searchRequestDTO.setSize(fullDataRequestDTO.getSize());

        // 당사자가 assignee 로 들어가 있는 경우
        SimpleQuery<SearchDocDTO> 요구사항_쿼리_목록 = 빌더에_넣을_공통_쿼리_목록_세팅(SimpleQuery.search(searchRequestDTO),fullDataRequestDTO);

        // 요구사항 목록 조회
        요구사항_쿼리_목록.andTermQueryMust("isReq", true);
        List<AlmIssueEntity> 요구사항_목록 = esCommonRepositoryWrapper.findRecentDocsByScrollApi(요구사항_쿼리_목록); // 스크롤API 필요

        // 하위이슈 목록 조회, ALM 직접 생성 연결이슈
        SimpleQuery<SearchDocDTO> 하위이슈_ALM직접연결이슈_쿼리_목록 = 빌더에_넣을_공통_쿼리_목록_세팅(SimpleQuery.search(searchRequestDTO), fullDataRequestDTO);
        하위이슈_ALM직접연결이슈_쿼리_목록.andTermQueryMust("isReq",false);
        List<AlmIssueEntity> 하위이슈_ALM직접연결이슈_목록 = esCommonRepositoryWrapper.findRecentDocsByScrollApi(하위이슈_ALM직접연결이슈_쿼리_목록); // 스크롤API 필요


        // for 반복문을 사용해 upperKey 여부에 따라, 하위이슈 연결이슈 분리
        for (AlmIssueEntity entity : 하위이슈_ALM직접연결이슈_목록) {
            String upperKey = entity.getUpperKey();
            하위이슈_연결이슈_docIdSet.add(entity.getRecentId());
            if (upperKey == null || upperKey.isEmpty()) {
                ALM_생성_연결이슈.add(entity);  // upperKey가 없으면
            } else {
                하위이슈_목록.add(entity);  // upperKey가 있으면
            }
        }

        // 요구사항 Key Set
        Set<String> parentReqKeySet = 요구사항_목록.stream()
                .map(AlmIssueEntity::getKey).collect(Collectors.toSet());

        // 하위이슈만 있어서, 부모 요구사항이 필요한 경우
        List<String> 중복제거한_하위이슈의_부모키 = new ArrayList<>(
                하위이슈_목록.stream()
                        .map(AlmIssueEntity::getParentReqKey)
                        .collect(Collectors.toSet()));

        List<String> 하위이슈만_존재하는경우의_부모키 = 중복제거한_하위이슈의_부모키.stream()
                .filter(Objects::nonNull) // null 제거
                .filter(key -> !parentReqKeySet.contains(key))
                .collect(Collectors.toList());

        if (!하위이슈만_존재하는경우의_부모키.isEmpty()) {
            하위이슈만_존재하는_이슈의_요구사항이슈_목록 = esCommonRepositoryWrapper.findRecentDocsByScrollApi(
                SimpleQuery.termQueryMust("isReq", true).andTermsQueryFilter("key",하위이슈만_존재하는경우의_부모키)
            );

        } else {
            log.info("하위이슈만 존재하는 경우 없음.");
        }

        for (AlmIssueEntity entity : 요구사항_목록) {
            List<String> linkedIssues = entity.getLinkedIssues();
            if (linkedIssues != null && !linkedIssues.isEmpty()) {
                // linkedIssues 배열을 Set에 추가하여 중복 제거
                for (String docId : linkedIssues) {
                    linkedIssuesDocIdSet.add(docId);
                }
            }
        }

        // 차집합 ( 요구사항의 연결이슈 배열 - 하위이슈_연결이슈_docIdSet) :: 이미 조회한 doc 제외
        linkedIssuesDocIdSet.removeAll(하위이슈_연결이슈_docIdSet);
        // 연결이슈만 존재하고, 연결이슈를 필요로 하는 요구사항이슈가 없는 경우
        List<String> linkedIssuesDocIdList = new ArrayList<>(linkedIssuesDocIdSet);
        List<AlmIssueEntity> 요구사항_연결이슈_목록 = esCommonRepositoryWrapper.findRecentDocsByScrollApi(
                SimpleQuery.termQueryMust("recent_id", linkedIssuesDocIdList).andTermQueryMust("isReq",false)
        ); // 스크롤API 필요

        모든_이슈_목록.addAll(요구사항_목록);
        if (!하위이슈만_존재하는_이슈의_요구사항이슈_목록.isEmpty()) {
            모든_이슈_목록.addAll(하위이슈만_존재하는_이슈의_요구사항이슈_목록);
        }
        모든_이슈_목록.addAll(하위이슈_목록);
        모든_이슈_목록.addAll(ALM_생성_연결이슈);
        if (!요구사항_연결이슈_목록.isEmpty()) {
            모든_이슈_목록.addAll(요구사항_연결이슈_목록);
        }
        log.info("[리포트_서비스_프로세스 :: 이슈_목록_가져오기2 ] :: 요구사항 이슈 총 갯수 => {}", 요구사항_목록.size() + 하위이슈만_존재하는_이슈의_요구사항이슈_목록.size());

        FullDataResponseDTO 결과 = new FullDataResponseDTO();
        결과.setTotalHits((long) 모든_이슈_목록.size());
        결과.setIssueEntityList(모든_이슈_목록);
        log.info("[리포트_서비스_프로세스 :: 이슈_목록_가져오기2 ] :: 이슈 총 갯수 => {}", 모든_이슈_목록.size());
        return 결과;
    }


    @Override
    public FullDataResponseDTO 이슈_목록_가져오기_calendar(FullDataRequestDTO fullDataRequestDTO) {

        List<AlmIssueEntity> 모든_이슈_목록 = new ArrayList<>();
        // upperKey 필드가 없는 지라이슈_엔티티를 담을 리스트
        List<AlmIssueEntity> ALM_생성_연결이슈 = new ArrayList<>();
        // upperKey 필드가 있는 지라이슈_엔티티를 담을 리스트
        List<AlmIssueEntity> 하위이슈_목록 = new ArrayList<>();
        List<AlmIssueEntity> 하위이슈만_존재하는_이슈의_요구사항이슈_목록 = new ArrayList<>();
        Set<String> 하위이슈_연결이슈_docIdSet = new HashSet<>();
        // 요구사항 이슈의 연결이슈 docIdSet
        Set<String> linkedIssuesDocIdSet = new HashSet<>();

        SearchRequestDTO searchRequestDTO = new SearchRequestDTO();
        searchRequestDTO.setPage(fullDataRequestDTO.getPage());
        searchRequestDTO.setSize(fullDataRequestDTO.getSize());

        // 당사자가 assignee 로 들어가 있는 경우
        SimpleQuery<SearchDocDTO> 공통_쿼리_목록 = 빌더에_넣을_공통_쿼리_목록_세팅(SimpleQuery.search(searchRequestDTO), fullDataRequestDTO);

        // 요구사항 목록 조회
        공통_쿼리_목록.andTermQueryMust("isReq", true);
        DocumentResultWrapper<AlmIssueEntity> docsBySearchAfter = esCommonRepositoryWrapper.findDocsBySearchAfter(공통_쿼리_목록);
        List<AlmIssueEntity> 요구사항_목록 = docsBySearchAfter.toDocs();

        // 하위이슈 목록 조회, ALM 직접 생성 연결이슈
        SimpleQuery<SearchDocDTO> 하위이슈_ALM직접연결이슈_쿼리_목록 = 빌더에_넣을_공통_쿼리_목록_세팅(SimpleQuery.search(searchRequestDTO), fullDataRequestDTO);
        하위이슈_ALM직접연결이슈_쿼리_목록.andTermQueryMust("isReq",false);

        List<AlmIssueEntity> 하위이슈_ALM직접연결이슈_목록 = esCommonRepositoryWrapper.findDocsBySearchAfter(하위이슈_ALM직접연결이슈_쿼리_목록).toDocs();

        // for 반복문을 사용해 upperKey 여부에 따라, 하위이슈 연결이슈 분리
        for (AlmIssueEntity entity : 하위이슈_ALM직접연결이슈_목록) {
            String upperKey = entity.getUpperKey();
            하위이슈_연결이슈_docIdSet.add(entity.getRecentId());
            if (upperKey == null || upperKey.isEmpty()) {
                ALM_생성_연결이슈.add(entity);  // upperKey가 없으면
            } else {
                하위이슈_목록.add(entity);  // upperKey가 있으면
            }
        }

        // 요구사항 Key Set
        Set<String> parentReqKeySet = 요구사항_목록.stream()
                .map(AlmIssueEntity::getKey).collect(Collectors.toSet());

        // 하위이슈만 있어서, 부모 요구사항이 필요한 경우
        List<String> 중복제거한_하위이슈의_부모키 = new ArrayList<>(
                하위이슈_목록.stream()
                        .map(AlmIssueEntity::getParentReqKey)
                        .collect(Collectors.toSet()));

        List<String> 하위이슈만_존재하는경우의_부모키 = 중복제거한_하위이슈의_부모키.stream()
                .filter(Objects::nonNull) // null 제거
                .filter(key -> !parentReqKeySet.contains(key))
                .collect(Collectors.toList());

        if (!하위이슈만_존재하는경우의_부모키.isEmpty()) {
            하위이슈만_존재하는_이슈의_요구사항이슈_목록
                = esCommonRepositoryWrapper.findDocsBySearchAfter(
                        SimpleQuery.termQueryMust("isReq", true)
                            .andTermsQueryFilter("key", 하위이슈만_존재하는경우의_부모키)).toDocs();

        } else {
            log.info("하위이슈만 존재하는 경우 없음.");
        }

        for (AlmIssueEntity entity : 요구사항_목록) {
            List<String> linkedIssues = entity.getLinkedIssues();
            if (linkedIssues != null && !linkedIssues.isEmpty()) {
                // linkedIssues 배열을 Set에 추가하여 중복 제거
                for (String docId : linkedIssues) {
                    linkedIssuesDocIdSet.add(docId);
                }
            }
        }

        // 차집합 ( 요구사항의 연결이슈 배열 - 하위이슈_연결이슈_docIdSet) :: 이미 조회한 doc 제외
        linkedIssuesDocIdSet.removeAll(하위이슈_연결이슈_docIdSet);
        // 연결이슈만 존재하고, 연결이슈를 필요로 하는 요구사항이슈가 없는 경우
        List<String> linkedIssuesDocIdList = new ArrayList<>(linkedIssuesDocIdSet);

        List<AlmIssueEntity> 요구사항_연결이슈_목록 = esCommonRepositoryWrapper.findDocsByScrollApi(
                SimpleQuery.search(searchRequestDTO)
                        .andTermsQueryFilter("recent_id", linkedIssuesDocIdList)
                        .andTermQueryMust("isReq",false)
        ); // 스크롤API 필요

        모든_이슈_목록.addAll(요구사항_목록);
        if (!하위이슈만_존재하는_이슈의_요구사항이슈_목록.isEmpty()) {
            모든_이슈_목록.addAll(하위이슈만_존재하는_이슈의_요구사항이슈_목록);
        }
        모든_이슈_목록.addAll(하위이슈_목록);
        모든_이슈_목록.addAll(ALM_생성_연결이슈);
        if (!요구사항_연결이슈_목록.isEmpty()) {
            모든_이슈_목록.addAll(요구사항_연결이슈_목록);
        }
        log.info("[리포트_서비스_프로세스 :: 이슈_목록_가져오기2 ] :: 요구사항 이슈 총 갯수 => {}", 요구사항_목록.size() + 하위이슈만_존재하는_이슈의_요구사항이슈_목록.size());
        log.info("[리포트_서비스_프로세스 :: 이슈_목록_가져오기2 ] :: 이슈 총 갯수 => {}", 모든_이슈_목록.size());
        List<AlmIssueEntity> issues = 모든_이슈_목록;// 기존 이슈 조회 로직

        // 조회된 데이터에서 중복 제거
        Map<String, AlmIssueEntity> uniqueIssues = issues.stream()
                .collect(Collectors.toMap(
                        issue -> issue.getKey() + "::" + issue.getUpdated(),
                        Function.identity(),
                        (existing, newer) -> newer,
                        LinkedHashMap::new
                ));

        List<AlmIssueEntity> distinctIssues = new ArrayList<>(uniqueIssues.values());
        FullDataResponseDTO 결과 = new FullDataResponseDTO();
        결과.setTotalHits((long) distinctIssues.size());
        결과.setIssueEntityList(distinctIssues);
        log.info("[리포트_서비스_프로세스 :: 이슈_목록_가져오기2 ] :: 엔진 이슈 중복제거 갯수 => {}", distinctIssues.size());
        return 결과;
    }

    @Override
    public FullDataResponseDTO getReqIssueList(FullDataRequestDTO fullDataRequestDTO){
        SearchRequestDTO searchRequestDTO = new SearchRequestDTO();
        searchRequestDTO.setPage(fullDataRequestDTO.getPage());
        searchRequestDTO.setSize(fullDataRequestDTO.getSize());

        SimpleQuery<SearchDocDTO> simpleQuery = 빌더에_넣을_공통_쿼리_목록_세팅(SimpleQuery.search(searchRequestDTO), fullDataRequestDTO);
        simpleQuery.andTermsQueryFilter("pdServiceVersionIds", fullDataRequestDTO.getPdServiceVersionIds())
                .andTermQueryMust("isReq", true);

        List<AlmIssueEntity> reqList = esCommonRepositoryWrapper.findRecentDocsByScrollApi(simpleQuery);

        FullDataResponseDTO result = new FullDataResponseDTO();
        result.setTotalHits((long) reqList.size());
        result.setIssueEntityList(reqList);
        return result;
    }

    @Override
    public AssigneeIssueCountVO getAssigneeAggregationList(FullDataRequestDTO fullDataRequestDTO) {

        DocumentAggregations documentAggregations = esCommonRepositoryWrapper.aggregateRecentDocs(
                SimpleQuery.aggregation(
                    AggregationRequestDTO.builder()
                        .mainField("assignee.assignee_emailAddress.keyword")
                        .mainFieldAlias("assigneeEmail")
                        .size(fullDataRequestDTO.getSize()).build())
                        .andTermsQueryFilter("pdServiceId", fullDataRequestDTO.getPdServiceIds())
                        .andTermsQueryFilter("pdServiceVersions", fullDataRequestDTO.getPdServiceVersionIds())
                        .andExistsQueryFilter("assignee"));

        List<DocumentBucket> documentBuckets = documentAggregations.deepestList();

        long totalIssueCount = documentAggregations.getTotalHits();
        long totalAssigneeCount = documentBuckets.size();

        Map<String, Long> assigneeCounts = new HashMap<>();
        for (DocumentBucket documentBucket : documentBuckets) {
            assigneeCounts.put(documentBucket.valueByName("assigneeEmail"), documentBucket.countByName("assigneeEmail"));
        }

        double avgIssuesPerAssignee = totalAssigneeCount > 0
                ? (double) totalIssueCount / totalAssigneeCount
                : 0.0;

        long maxIssues = assigneeCounts.values().stream()
                .mapToLong(Long::longValue)
                .max()
                .orElse(0L);

        long minIssues = assigneeCounts.values().stream()
                .mapToLong(Long::longValue)
                .min()
                .orElse(0L);

        return new AssigneeIssueCountVO(
                totalIssueCount,
                totalAssigneeCount,
                avgIssuesPerAssignee,
                maxIssues,
                minIssues,
                assigneeCounts
        );
    }


}
