package com.arms.egovframework.javaservice.gcframework.provider;

import com.arms.config.GiteaUserConfig;
import com.arms.egovframework.javaservice.gcframework.model.FileContent;
import com.arms.egovframework.javaservice.gcframework.model.GitFileInfo;
import com.arms.egovframework.javaservice.gcframework.model.RepoType;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.util.Base64;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * GitRepositoryProvider 인터페이스의 구현체로,
 * 실제 Gitea REST API와 통신하여 Git 작업(listDirectory, getFileContent 등)을 수행합니다.
 * GiteaUserConfig 로부터 접속 정보를 주입받아 사용합니다.
 * */

@Service
@Slf4j
public class GiteaRepositoryProvider implements GitRepositoryProvider {

    private final GiteaUserConfig giteaUserConfig;
    private final ObjectMapper objectMapper;

    public GiteaRepositoryProvider(GiteaUserConfig giteaUserConfig, ObjectMapper objectMapper) {
        this.giteaUserConfig = giteaUserConfig;
        this.objectMapper = objectMapper;
    }

    @Override
    public RepoType getType() {
        return RepoType.GITEA;
    }


    /**
     * Gitea API 호출을 위한 기본 URL을 생성합니다.
     * 예: {baseUrl}/api/v1/repos/{owner}/{repository-name}/contents
     */
    private String getBaseApiUrl(String owner, String repoName) {
        return String.format("%s/api/v1/repos/%s/%s/contents",
                giteaUserConfig.getBaseUrl(),
                owner,
                repoName);
    }

    /**
     * HttpURLConnection을 설정하고 인증 헤더를 추가합니다.
     */
    @SuppressWarnings("java:S2647")
    private HttpURLConnection createAuthenticatedConnection(String requestUrl, String method) throws Exception {
        URL url = new URL(requestUrl);
        HttpURLConnection connection = (HttpURLConnection) url.openConnection();

        String auth = giteaUserConfig.getGitUsername() + ":" + giteaUserConfig.getGitPassword();
        // Base64 인코딩 시 URl-safe 인코더가 아닌 일반 인코더 사용 (RFC 4648 Section 4)
        String encodedAuth = Base64.getEncoder().encodeToString(auth.getBytes(StandardCharsets.UTF_8));

        connection.setRequestMethod(method);
        connection.setRequestProperty("Authorization", "Basic " + encodedAuth);
        connection.setRequestProperty("Accept", "application/json"); // JSON 응답을 기대

        connection.setDoOutput(false); // 기본은 GET/HEAD에 대한 false
        if ("POST".equals(method) || "PUT".equals(method) || "DELETE".equals(method)) {
            connection.setDoOutput(true); // PUT/POST/DELETE는 Body를 보낼 수 있도록 설정
            connection.setRequestProperty("Content-Type", "application/json");
        }

        return connection;
    }

    @Override
    public List<GitFileInfo> getListFilesAndDirectories(String owner, String repoName, String directoryPath, String branch) {

        String requestUrl = String.format("%s/%s?ref=%s", getBaseApiUrl(owner, repoName), directoryPath, branch);
        log.info("[GiteaRepositoryProvider] Listing directory:: requestUrl => {}", requestUrl);

        try {
            HttpURLConnection connection = createAuthenticatedConnection(requestUrl, "GET");

            List<GitFileInfo> fileInfos = readJsonResponse(connection, new TypeReference<List<GitFileInfo>>() {});

            if (fileInfos != null) {
                fileInfos.forEach(info -> {
                    if (info.getUrl() != null) {
                        info.setUrl(rewriteUrlToInternal(info.getUrl()));
                    }
                });
            }
            return fileInfos;

        } catch (Exception e) {
            log.error("Failed to get File list of Gitea repository -> repoName => {}: {}", repoName, e.getMessage(), e);
            throw new RuntimeException("Gitea repository file listing is failed: " + e.getMessage(), e);
        }
    }

    @Override
    public FileContent getFileContent(String owner, String repoName, String branch, String filePath) {
        // Gitea API: GET /repos/{owner}/{repo}/contents/{path}?ref={branch}
        String requestUrl = String.format("%s/%s?ref=%s", getBaseApiUrl(owner, repoName), filePath, branch);
        log.info("[GiteaRepositoryProvider] Getting file content: {}", requestUrl);

        try {
            HttpURLConnection connection = createAuthenticatedConnection(requestUrl, "GET");
            // Gitea API 응답은 Map<String, Object> 형태로 받아 세부 필드를 FileContent에 매핑합니다.
            // ObjectMapper는 @JsonProperty가 없는 필드에 대해 기본적으로 camelCase to snake_case 매핑을 시도합니다.
            Map<String, Object> response = readJsonResponse(connection, new TypeReference<Map<String, Object>>() {});

            if (response != null && response.containsKey("content")) {
                FileContent fileContent = new FileContent();
                fileContent.setFileName((String) response.get("name")); // Gitea 응답의 'name'
                fileContent.setFilePath((String) response.get("path")); // Gitea 응답의 'path'
                fileContent.setContent((String) response.get("content")); // Base64 인코딩된 상태
                fileContent.setEncoding((String) response.get("encoding")); // Gitea 응답의 'encoding'
                fileContent.setSha((String) response.get("sha")); // Gitea 응답의 'sha'

                // Gitea 응답의 'size'는 Number 타입일 수 있으므로 long으로 변환
                Object sizeObj = response.get("size");
                if (sizeObj instanceof Number) {
                    fileContent.setSize(((Number) sizeObj).longValue());
                }

                return fileContent;
            }
            return null;
        } catch (Exception e) {
            log.error("Failed to get Gitea file content {}: {}", filePath, e.getMessage(), e);
            throw new RuntimeException("Gitea file content retrieval failed: " + e.getMessage(), e);
        }
    }

    @Override
    public boolean upsertFile(String owner, String repoName, String branch, String filePath, String content, String commitMessage) {
        // Gitea API: PUT /repos/{owner}/{repo}/contents/{path}
        // Body: { "message": "commit message", "content": "base64_encoded_content", "branch": "branch_name", "sha": "optional_sha_for_update" }
        String requestUrl = String.format("%s/%s", getBaseApiUrl(owner, repoName), filePath);
        log.info("[GiteaRepositoryProvider] Upserting file: {}", requestUrl);

        try {
            HttpURLConnection connection = createAuthenticatedConnection(requestUrl, "PUT");

            Map<String, Object> requestBody = Map.of(
                    "message", commitMessage,
                    "content", Base64.getEncoder().encodeToString(content.getBytes(StandardCharsets.UTF_8)), // Gitea는 Base64 인코딩된 내용을 요구
                    "branch", branch
                    // "sha": 기존 파일 업데이트 시 SHA 필요. 새 파일 생성 시에는 불필요.
                    // 이 예시에서는 SHA를 넘기지 않아도 대부분의 Gitea 버전에서 작동하지만,
                    // 정확한 업데이트를 위해서는 기존 파일의 SHA를 조회해서 함께 넘겨주는 것이 안전합니다.
                    // 만약 sha가 없으면, Gitea는 해당 경로에 파일이 없으면 생성, 있으면 덮어쓰기를 시도합니다.
            );
            writeRequestBody(connection, requestBody);

            int responseCode = connection.getResponseCode();
            if (responseCode == HttpURLConnection.HTTP_OK || responseCode == HttpURLConnection.HTTP_CREATED) {
                log.info("Successfully upserted Gitea file: {}", filePath);
                return true;
            } else {
                String errorMessage = readErrorStream(connection);
                log.error("Failed to upsert Gitea file {}. HTTP Error {}: {}", filePath, responseCode, errorMessage);
                throw new RuntimeException("Gitea file upsert failed: " + errorMessage + " (Code: " + responseCode + ")");
            }

        } catch (Exception e) {
            log.error("Failed to upsert Gitea file {}: {}", filePath, e.getMessage(), e);
            throw new RuntimeException("Gitea file upsert failed: " + e.getMessage(), e);
        }
    }

    @Override
    public boolean deleteFile(String owner, String repoName, String branch, String filePath, String commitMessage) {
        // Gitea API: DELETE /repos/{owner}/{repo}/contents/{path}
        // Body: { "message": "commit message", "branch": "branch_name", "sha": "file_sha" }
        // 파일을 삭제하려면 반드시 파일의 SHA 값이 필요합니다.
        String requestUrl = String.format("%s/%s", getBaseApiUrl(owner, repoName), filePath);
        log.info("[GiteaRepositoryProvider] Deleting file: {}", requestUrl);

        try {
            // 삭제를 위해 SHA 값을 먼저 조회해야 합니다.
            FileContent existingFile = getFileContent(owner, repoName, branch, filePath);
            if (existingFile == null || existingFile.getSha() == null) {
                log.warn("File {} not found or SHA missing for deletion. Cannot proceed with delete.", filePath);
                // 파일이 없거나 SHA를 알 수 없으면 삭제는 실패로 간주 (또는 이미 삭제된 것으로 간주)
                return false;
            }
            String fileSha = existingFile.getSha();

            HttpURLConnection connection = createAuthenticatedConnection(requestUrl, "DELETE");

            Map<String, Object> requestBody = Map.of(
                    "message", commitMessage,
                    "branch", branch,
                    "sha", fileSha // Gitea 삭제 시 필수
            );
            writeRequestBody(connection, requestBody);

            int responseCode = connection.getResponseCode();
            if (responseCode == HttpURLConnection.HTTP_NO_CONTENT) { // 204 No Content
                log.info("Successfully deleted Gitea file: {}", filePath);
                return true;
            } else {
                String errorMessage = readErrorStream(connection);
                log.error("Failed to delete Gitea file {}. HTTP Error {}: {}", filePath, responseCode, errorMessage);
                throw new RuntimeException("Gitea file deletion failed: " + errorMessage + " (Code: " + responseCode + ")");
            }

        } catch (Exception e) {
            log.error("Failed to delete Gitea file {}: {}", filePath, e.getMessage(), e);
            throw new RuntimeException("Gitea file deletion failed: " + e.getMessage(), e);
        }
    }



    // 이거 기술 잘못되었음.
    /**
     * Gitea의 download_url이 실제 다운로드 가능한 URL이 아닐 수 있으므로 재작성합니다.
     * Gitea는 때때로 `/raw/branch/` 경로가 아닌 `/src/branch/` 경로를 다운로드 URL로 제공할 수 있습니다.
     * 이를 실제 Raw 파일 다운로드 URL로 변경합니다.
     * 예: `http://localhost:3000/myuser/my-repo/src/branch/main/path/to/file.json`
     * -> `http://localhost:3000/myuser/my-repo/raw/branch/main/path/to/file.json`
     */
    private String rewriteGiteaUrl(String downloadUrl) {
        if (downloadUrl != null && downloadUrl.contains("/src/branch/")) {
            return downloadUrl.replace("/src/branch/", "/raw/branch/");
        }
        return downloadUrl;
    }

    /**
     * Gitea의 url은 외부에서 접근하는 URL path 형태로 저장됨.
     * Gitea cluster 내부 통신을 위한 path 로 수정
     */
    private String rewriteUrlToInternal(String url) {
        // 정규표현식: 프로토콜(http 또는 https)부터 /gitea까지 매칭
        String regex = "(https?://)([^/]+)(/gitea)";
        // 기존 URL에서 정규표현식을 사용해 부분 문자열 치환
        return url.replaceAll(regex, giteaUserConfig.getReplaceUrl());
    }

    /**
     * HttpURLConnection 응답을 읽어 JSON 객체 또는 리스트로 파싱합니다.
     */
    private <T> T readJsonResponse(HttpURLConnection connection, TypeReference<T> typeRef) throws Exception {
        int responseCode = connection.getResponseCode();
        if (responseCode != HttpURLConnection.HTTP_OK &&
                responseCode != HttpURLConnection.HTTP_CREATED && // PUT/POST 성공 시 201
                responseCode != HttpURLConnection.HTTP_NO_CONTENT // DELETE 성공 시 204
        ) {
            String errorMessage = readErrorStream(connection);
            log.error("HTTP Error {}: {}", responseCode, errorMessage);
            throw new RuntimeException("Failed to process Gitea request. Response Code: " + responseCode + ", Message: " + errorMessage);
        }

        // DELETE 요청은 HTTP_NO_CONTENT(204)를 반환하며, 응답 본문이 없을 수 있습니다.
        if (responseCode == HttpURLConnection.HTTP_NO_CONTENT) {
            return null; // 본문이 없는 경우 null 반환
        }

        try (BufferedReader reader = new BufferedReader(new InputStreamReader(connection.getInputStream(), StandardCharsets.UTF_8))) {
            return objectMapper.readValue(reader, typeRef);
        }
    }

    /**
     * 에러 스트림을 읽어 에러 메시지를 반환합니다.
     */
    private String readErrorStream(HttpURLConnection connection) {
        try (BufferedReader reader = new BufferedReader(new InputStreamReader(connection.getErrorStream(), StandardCharsets.UTF_8))) {
            if (reader != null) {
                return reader.lines().collect(Collectors.joining("\n"));
            }
            return "No error message available from stream.";
        } catch (Exception e) {
            log.error("Failed to read error stream", e);
            return "Unknown error while reading error stream: " + e.getMessage();
        }
    }

    /**
     * HttpURLConnection에 JSON 요청 바디를 씁니다.
     */
    private void writeRequestBody(HttpURLConnection connection, Map<String, Object> body) throws Exception {
        try (OutputStream os = connection.getOutputStream()) {
            os.write(objectMapper.writeValueAsBytes(body));
            os.flush();
        }
    }
}
