/*
 * Decompiled with CFR 0.152.
 */
package com.arms.api.wiki.service;

import com.arms.api.wiki.dto.SelectionInfo;
import com.arms.api.wiki.dto.UserInfoDTO;
import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.HashOperations;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;

@Service
public class SessionRegistryService {
    private static final Logger logger = Logger.getLogger(SessionRegistryService.class.getName());
    private static final String SESSION_USERS_KEY_PREFIX = "session:users:";
    private static final long SESSION_EXPIRY_MINUTES = 60L;
    private final RedisTemplate<String, Object> redisTemplate;
    private final HashOperations<String, String, UserInfoDTO> hashOperations;

    @Autowired
    public SessionRegistryService(RedisTemplate<String, Object> redisTemplate) {
        this.redisTemplate = redisTemplate;
        this.hashOperations = redisTemplate.opsForHash();
    }

    private String getSessionDocumentKey(String sessionId, String documentId) {
        return SESSION_USERS_KEY_PREFIX + sessionId + ":" + documentId;
    }

    private void touchKey(String key) {
        this.redisTemplate.expire((Object)key, 60L, TimeUnit.MINUTES);
    }

    public void userJoined(String sessionId, String documentId, UserInfoDTO userInfo) {
        if (sessionId == null || documentId == null || userInfo == null || userInfo.getId() == null) {
            logger.warning("Attempted to add a user with null sessionId, documentId, info, or user ID.");
            return;
        }
        String key = this.getSessionDocumentKey(sessionId, documentId);
        String userId = userInfo.getId();
        try {
            this.hashOperations.put((Object)key, (Object)userId, (Object)userInfo);
            this.touchKey(key);
            logger.info(String.format("[Session: %s] User [%s] (%s) joined/updated in Redis for document [%s]. Key: %s", sessionId, userId, userInfo.getName(), documentId, key));
        }
        catch (Exception e) {
            logger.severe(String.format("Redis error adding user [%s] to key [%s]: %s", userId, key, e.getMessage()));
        }
    }

    public boolean userLeftDocument(String sessionId, String documentId, String userId) {
        if (sessionId == null || documentId == null || userId == null) {
            logger.warning("Attempted to remove a user with null sessionId, documentId, or userId.");
            return false;
        }
        String key = this.getSessionDocumentKey(sessionId, documentId);
        boolean removed = false;
        try {
            if (this.hashOperations.delete((Object)key, new Object[]{userId}) > 0L) {
                logger.info(String.format("[Session: %s] User [%s] removed from Redis for document [%s]. Key: %s", sessionId, userId, documentId, key));
                removed = true;
                if (this.hashOperations.size((Object)key) == 0L) {
                    this.redisTemplate.delete((Object)key);
                    logger.info(String.format("[Session: %s] Redis key [%s] deleted as it became empty after user [%s] left.", sessionId, key, userId));
                } else {
                    this.touchKey(key);
                }
            } else {
                logger.fine(String.format("[Session: %s] Attempted to remove user [%s] from key [%s], but they were not found.", sessionId, userId, key));
            }
        }
        catch (Exception e) {
            logger.severe(String.format("Redis error removing user [%s] from key [%s]: %s", userId, key, e.getMessage()));
        }
        return removed;
    }

    public void updateUserState(String sessionId, String documentId, String userId, Map<String, Integer> cursorPosition, SelectionInfo selection) {
        if (sessionId == null || documentId == null || userId == null) {
            logger.warning("Cannot update state with null sessionId, documentId or userId.");
            return;
        }
        String key = this.getSessionDocumentKey(sessionId, documentId);
        try {
            UserInfoDTO user = (UserInfoDTO)this.hashOperations.get((Object)key, (Object)userId);
            if (user != null) {
                user.setCursorPosition(cursorPosition);
                user.setSelection(selection);
                this.hashOperations.put((Object)key, (Object)userId, (Object)user);
                this.touchKey(key);
                logger.finest(String.format("[Session: %s] Updated Redis state for user [%s] in doc [%s]. Key: %s", sessionId, userId, documentId, key));
            } else {
                logger.warning(String.format("[Session: %s] Cannot update state for user [%s], not found in Redis key [%s]", sessionId, userId, key));
            }
        }
        catch (Exception e) {
            logger.severe(String.format("Redis error updating state for user [%s] in key [%s]: %s", userId, key, e.getMessage()));
        }
    }

    public List<UserInfoDTO> getActiveParticipantsForDocument(String sessionId, String documentId, String requestingUserId) {
        List<UserInfoDTO> participants;
        block7: {
            if (sessionId == null || documentId == null) {
                logger.warning("Cannot get participants for null sessionId or documentId.");
                return Collections.emptyList();
            }
            String key = this.getSessionDocumentKey(sessionId, documentId);
            participants = Collections.emptyList();
            try {
                if (Boolean.TRUE.equals(this.redisTemplate.hasKey((Object)key))) {
                    Map usersInDocument = this.hashOperations.entries((Object)key);
                    logger.info(String.format("[Session: %s, Doc: %s] Fetched %d entries from Redis hash key [%s]. Keys: %s", sessionId, documentId, usersInDocument != null ? usersInDocument.size() : 0, key, usersInDocument != null ? usersInDocument.keySet() : "null"));
                    if (usersInDocument != null && !usersInDocument.isEmpty()) {
                        try {
                            participants = usersInDocument.entrySet().stream().filter(entry -> requestingUserId == null || !((String)entry.getKey()).equals(requestingUserId)).map(Map.Entry::getValue).collect(Collectors.toList());
                            logger.info(String.format("[Session: %s, Doc: %s] Successfully mapped entries to %d participants (excluding user [%s]). Key: %s", sessionId, documentId, participants.size(), requestingUserId, key));
                        }
                        catch (Exception e) {
                            logger.log(Level.SEVERE, String.format("[Session: %s, Doc: %s] Error during stream processing of Redis entries for key [%s]: %s", sessionId, documentId, key, e.getMessage()), e);
                            participants = Collections.emptyList();
                        }
                        this.touchKey(key);
                        break block7;
                    }
                    logger.fine(String.format("Redis key not found for session/document: %s", key));
                    break block7;
                }
                logger.fine(String.format("Redis key not found for session/document: %s", key));
            }
            catch (Exception e) {
                logger.severe(String.format("Redis error getting participants for key [%s]: %s", key, e.getMessage()));
                participants = Collections.emptyList();
            }
        }
        logger.info(String.format("[Session: %s, Doc: %s] Returning %d participants.", sessionId, documentId, participants.size()));
        return participants;
    }

    public List<Map.Entry<String, String>> userLeftAllSessions(String userId) {
        if (userId == null) {
            logger.warning("Attempted to remove a user with null ID.");
            return Collections.emptyList();
        }
        ArrayList<Map.Entry<String, String>> affectedEntries = new ArrayList<Map.Entry<String, String>>();
        String pattern = "session:users:*";
        logger.info(String.format("Scanning Redis keys with pattern '%s' to remove user [%s]...", pattern, userId));
        try {
            Set keys = this.redisTemplate.keys((Object)pattern);
            if (keys != null) {
                for (String key : keys) {
                    try {
                        String[] parts = key.substring(SESSION_USERS_KEY_PREFIX.length()).split(":", 2);
                        if (parts.length == 2) {
                            String sessionId = parts[0];
                            String documentId = parts[1];
                            if (this.hashOperations.delete((Object)key, new Object[]{userId}) <= 0L) continue;
                            logger.info(String.format("[Session: %s] User [%s] removed from Redis document [%s]. Key: %s", sessionId, userId, documentId, key));
                            affectedEntries.add(new AbstractMap.SimpleEntry<String, String>(sessionId, documentId));
                            if (this.hashOperations.size((Object)key) != 0L) continue;
                            this.redisTemplate.delete((Object)key);
                            logger.info(String.format("[Session: %s] Redis key [%s] deleted as it became empty after user [%s] left.", sessionId, key, userId));
                            continue;
                        }
                        logger.warning("Could not parse sessionId and documentId from key: " + key);
                    }
                    catch (Exception e) {
                        logger.severe(String.format("Error processing key [%s] while removing user [%s]: %s", key, userId, e.getMessage()));
                    }
                }
            } else {
                logger.warning("Redis keys command returned null for pattern: " + pattern);
            }
        }
        catch (Exception e) {
            logger.severe(String.format("Redis error during key scanning for user [%s] removal: %s", userId, e.getMessage()));
        }
        if (affectedEntries.isEmpty()) {
            logger.fine("User [" + userId + "] was not found in any active Redis session/document key during scan.");
        } else {
            logger.info(String.format("User [%s] removed from %d session/document entries in Redis.", userId, affectedEntries.size()));
        }
        return affectedEntries;
    }
}

