package com.arms.api.serverinfo.controller;

import com.arms.api.serverinfo.model.ServerInfo;
import com.arms.api.serverinfo.model.ServerInfoEntity;
import com.arms.api.serverinfo.service.ServerInfoService;
import com.arms.api.util.errors.ErrorCode;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import java.util.List;
import java.util.Map;

@Slf4j
@RestController
@RequestMapping("/engine/serverInfo")
public class ServerInfoController {

    @Autowired
    private ServerInfoService serverInfoService;

    @ResponseBody
    @RequestMapping(
            value = {""},
            method = {RequestMethod.POST}
    )
    public ServerInfoEntity saveOrUpdateServerInfo(@RequestBody ServerInfo serverInfo) {

        if (serverInfo == null) {
            throw new IllegalArgumentException(ErrorCode.SERVER_INFO_ERROR.getErrorMsg());
        }
        else if (serverInfo.getConnectId() == null) {
            throw new IllegalArgumentException(ErrorCode.SERVER_ID_INFO_ERROR.getErrorMsg());
        }
        else if (StringUtils.isEmpty(serverInfo.getUri())) {
            throw new IllegalArgumentException(ErrorCode.SERVER_URI_INFO_ERROR.getErrorMsg());
        }
        else if (StringUtils.isEmpty(serverInfo.getUserId())) {
            throw new IllegalArgumentException(ErrorCode.SERVER_ID_INFO_ERROR.getErrorMsg());
        }
        else if (StringUtils.isEmpty(serverInfoService.getDecryptPasswordOrToken(serverInfo))) {
            throw new IllegalArgumentException(ErrorCode.SERVER_PW_OR_API_TOKEN_INFO_ERROR.getErrorMsg());
        }
        else if (StringUtils.isEmpty(serverInfo.getType())) {
            throw new IllegalArgumentException(ErrorCode.SERVER_TYPE_INFO_ERROR.getErrorMsg());
        }

        log.info("[ServerInfoController :: saveOrUpdateServerInfo] :: {}",serverInfo);

        return serverInfoService.saveOrUpdateServerInfo(serverInfo);
    }

    @GetMapping("/listByConnectIds")
    public List<ServerInfo> serverInfoListByConnectIds(@RequestParam("connectIds") List<String> connectIds) {
        log.info("[ServerInfoController :: serverInfoListByConnectIds] :: {}", connectIds.toString());

        return serverInfoService.serverInfoListByConnectIds(connectIds);
    }


    @PostMapping(value = {"/backup/scheduler"})
    public void serverInfoBackup() {
        serverInfoService.serverInfoBackup();
    }

    /*
    * 삭제 관련 차후 설계 후 개발 진행
    * */
    @ResponseBody
    @RequestMapping(
            value = {""},
            method = {RequestMethod.DELETE}
    )
    public ServerInfo deleteServerInfo(@RequestBody ServerInfo serverInfo) throws Exception {

        if (serverInfo == null || serverInfo.getConnectId() == null) {
            throw new Exception("삭제 요청 데이터가 없습니다.");
        }

        log.info("[ServerInfoController :: deleteServerInfo] :: {}", serverInfo.getConnectId());

        serverInfoService.deleteServerInfo(serverInfo.getConnectId());

        return serverInfo;

    }

    @ResponseBody
    @RequestMapping(
            value = {"/serverTypeMap"},
            method = {RequestMethod.GET}
    )
    public ResponseEntity<Map<String, String>> getServerTypeMap() {

        log.info("[ServerInfoController :: getServerTypeMap]");

        Map<String, String> serverTypeMap = serverInfoService.getServerIdToTypeMap();
        return ResponseEntity.ok(serverTypeMap);
    }
}
