package com.arms.api.report.export_service.generator;

import com.arms.api.report.export_service.model.ChartData;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import javax.imageio.ImageIO;
import java.awt.*;
import java.awt.image.BufferedImage;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.List;

/**
 * 차트 이미지 생성기
 *
 * <p>ChartData를 기반으로 PNG 이미지를 생성합니다.
 *
 * <p>참고1: 정교한 차트가 필요하면 JFreeChart 사용을 권장합니다.
 * <p>참고2: 25.12.03 현재 Frontend 에서 rendering 이미지 자체를 넘기는 것을 기본으로 진행.
 *
 * @author HS.Yang
 * @since 25.12.03
 */
@Slf4j
@Component
public class ChartImageGenerator {

    /** 기본 색상 팔레트 */
    private static final Color[] DEFAULT_COLORS = {
            new Color(66, 133, 244),   // Blue
            new Color(234, 67, 53),    // Red
            new Color(251, 188, 5),    // Yellow
            new Color(52, 168, 83),    // Green
            new Color(155, 89, 182),   // Purple
            new Color(52, 73, 94),     // Dark Gray
            new Color(230, 126, 34),   // Orange
            new Color(26, 188, 156)    // Teal
    };

    /**
     * 차트 이미지 생성
     */
    public byte[] generateChartImage(ChartData chartData) {
        int width = chartData.getOptions().getWidth();
        int height = chartData.getOptions().getHeight();

        BufferedImage image = new BufferedImage(width, height, BufferedImage.TYPE_INT_ARGB);
        Graphics2D g2d = image.createGraphics();

        // 안티앨리어싱
        g2d.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
        g2d.setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING, RenderingHints.VALUE_TEXT_ANTIALIAS_ON);

        // 배경
        g2d.setColor(Color.WHITE);
        g2d.fillRect(0, 0, width, height);

        try {
            switch (chartData.getType()) {
                case BAR:
                    drawBarChart(g2d, chartData, width, height);
                    break;
                case LINE:
                    drawLineChart(g2d, chartData, width, height);
                    break;
                case PIE:
                    drawPieChart(g2d, chartData, width, height);
                    break;
                default:
                    drawBarChart(g2d, chartData, width, height);
            }

            if (chartData.getOptions().isShowLegend()) {
                drawLegend(g2d, chartData, width, height);
            }

            return toByteArray(image);

        } catch (Exception e) {
            log.error("Failed to generate chart image", e);
            return createErrorImage(width, height);
        } finally {
            g2d.dispose();
        }
    }

    /**
     * 막대 차트
     */
    private void drawBarChart(Graphics2D g2d, ChartData chartData, int width, int height) {
        int margin = 60;
        int chartWidth = width - margin * 2;
        int chartHeight = height - margin * 2 - 30;

        List<String> labels = chartData.getLabels();
        List<ChartData.DataSeries> seriesList = chartData.getSeries();

        if (labels.isEmpty() || seriesList.isEmpty()) {
            return;
        }

        double maxValue = seriesList.stream()
                .flatMap(s -> s.getValues().stream())
                .max(Double::compareTo)
                .orElse(100.0);

        int numBars = labels.size();
        int numSeries = seriesList.size();
        int groupWidth = chartWidth / numBars;
        int barWidth = (groupWidth - 10) / numSeries;

        // 축
        g2d.setColor(Color.BLACK);
        g2d.drawLine(margin, margin, margin, margin + chartHeight);
        g2d.drawLine(margin, margin + chartHeight, margin + chartWidth, margin + chartHeight);

        // 바
        for (int i = 0; i < numBars; i++) {
            int groupX = margin + i * groupWidth + 5;

            for (int s = 0; s < numSeries; s++) {
                ChartData.DataSeries series = seriesList.get(s);
                double value = i < series.getValues().size() ? series.getValues().get(i) : 0;

                int barHeight = (int) ((value / maxValue) * chartHeight);
                int barX = groupX + s * barWidth;
                int barY = margin + chartHeight - barHeight;

                g2d.setColor(getSeriesColor(s, series.getColor()));
                g2d.fillRect(barX, barY, barWidth - 2, barHeight);

                if (chartData.getOptions().isShowValues()) {
                    g2d.setColor(Color.BLACK);
                    g2d.setFont(new Font("SansSerif", Font.PLAIN, 10));
                    g2d.drawString(String.format("%.0f", value), barX + barWidth / 4, barY - 5);
                }
            }

            // X축 라벨
            g2d.setColor(Color.BLACK);
            g2d.setFont(new Font("SansSerif", Font.PLAIN, 11));
            String label = labels.get(i);
            int labelWidth = g2d.getFontMetrics().stringWidth(label);
            g2d.drawString(label, groupX + groupWidth / 2 - labelWidth / 2 - 5, margin + chartHeight + 20);
        }
    }

    /**
     * 선 차트
     */
    private void drawLineChart(Graphics2D g2d, ChartData chartData, int width, int height) {
        int margin = 60;
        int chartWidth = width - margin * 2;
        int chartHeight = height - margin * 2 - 30;

        List<String> labels = chartData.getLabels();
        List<ChartData.DataSeries> seriesList = chartData.getSeries();

        if (labels.isEmpty() || seriesList.isEmpty()) {
            return;
        }

        double maxValue = seriesList.stream()
                .flatMap(s -> s.getValues().stream())
                .max(Double::compareTo)
                .orElse(100.0);

        int numPoints = labels.size();
        int pointGap = numPoints > 1 ? chartWidth / (numPoints - 1) : chartWidth;

        // 축
        g2d.setColor(Color.BLACK);
        g2d.drawLine(margin, margin, margin, margin + chartHeight);
        g2d.drawLine(margin, margin + chartHeight, margin + chartWidth, margin + chartHeight);

        // 시리즈
        for (int s = 0; s < seriesList.size(); s++) {
            ChartData.DataSeries series = seriesList.get(s);
            g2d.setColor(getSeriesColor(s, series.getColor()));
            g2d.setStroke(new BasicStroke(2));

            int[] xPoints = new int[numPoints];
            int[] yPoints = new int[numPoints];

            for (int i = 0; i < numPoints; i++) {
                double value = i < series.getValues().size() ? series.getValues().get(i) : 0;
                xPoints[i] = margin + i * pointGap;
                yPoints[i] = margin + chartHeight - (int) ((value / maxValue) * chartHeight);
            }

            g2d.drawPolyline(xPoints, yPoints, numPoints);

            for (int i = 0; i < numPoints; i++) {
                g2d.fillOval(xPoints[i] - 4, yPoints[i] - 4, 8, 8);
            }
        }

        // X축 라벨
        g2d.setColor(Color.BLACK);
        g2d.setFont(new Font("SansSerif", Font.PLAIN, 11));
        for (int i = 0; i < numPoints; i++) {
            String label = labels.get(i);
            int labelWidth = g2d.getFontMetrics().stringWidth(label);
            g2d.drawString(label, margin + i * pointGap - labelWidth / 2, margin + chartHeight + 20);
        }
    }

    /**
     * 파이 차트
     */
    private void drawPieChart(Graphics2D g2d, ChartData chartData, int width, int height) {
        int centerX = width / 2;
        int centerY = height / 2;
        int radius = Math.min(width, height) / 3;

        List<ChartData.DataSeries> seriesList = chartData.getSeries();
        if (seriesList.isEmpty() || seriesList.get(0).getValues().isEmpty()) {
            return;
        }

        List<Double> values = seriesList.get(0).getValues();
        double total = values.stream().mapToDouble(Double::doubleValue).sum();

        int startAngle = 0;
        for (int i = 0; i < values.size(); i++) {
            double value = values.get(i);
            int arcAngle = (int) Math.round((value / total) * 360);

            g2d.setColor(getSeriesColor(i, null));
            g2d.fillArc(centerX - radius, centerY - radius, radius * 2, radius * 2, startAngle, arcAngle);

            startAngle += arcAngle;
        }

        // 테두리
        g2d.setColor(Color.WHITE);
        g2d.setStroke(new BasicStroke(2));
        startAngle = 0;
        for (Double value : values) {
            int arcAngle = (int) Math.round((value / total) * 360);
            g2d.drawArc(centerX - radius, centerY - radius, radius * 2, radius * 2, startAngle, arcAngle);
            startAngle += arcAngle;
        }
    }

    /**
     * 범례
     */
    private void drawLegend(Graphics2D g2d, ChartData chartData, int width, int height) {
        List<ChartData.DataSeries> seriesList = chartData.getSeries();
        if (seriesList.isEmpty()) {
            return;
        }

        int legendX = width - 150;
        int legendY = 20;
        int itemHeight = 20;

        for (int i = 0; i < seriesList.size(); i++) {
            ChartData.DataSeries series = seriesList.get(i);

            g2d.setColor(getSeriesColor(i, series.getColor()));
            g2d.fillRect(legendX, legendY + i * itemHeight, 15, 15);

            g2d.setColor(Color.BLACK);
            g2d.setFont(new Font("SansSerif", Font.PLAIN, 11));
            String name = series.getName() != null ? series.getName() : "Series " + (i + 1);
            g2d.drawString(name, legendX + 20, legendY + i * itemHeight + 12);
        }
    }

    private Color getSeriesColor(int index, String colorHex) {
        if (colorHex != null && !colorHex.isEmpty()) {
            try {
                return Color.decode(colorHex);
            } catch (NumberFormatException e) {
                log.warn("Invalid color hex: {}", colorHex);
            }
        }
        return DEFAULT_COLORS[index % DEFAULT_COLORS.length];
    }

    private byte[] toByteArray(BufferedImage image) {
        try (ByteArrayOutputStream out = new ByteArrayOutputStream()) {
            ImageIO.write(image, "PNG", out);
            return out.toByteArray();
        } catch (IOException e) {
            throw new RuntimeException("Failed to convert image to byte array", e);
        }
    }

    private byte[] createErrorImage(int width, int height) {
        BufferedImage image = new BufferedImage(width, height, BufferedImage.TYPE_INT_ARGB);
        Graphics2D g2d = image.createGraphics();
        g2d.setColor(Color.LIGHT_GRAY);
        g2d.fillRect(0, 0, width, height);
        g2d.setColor(Color.RED);
        g2d.setFont(new Font("SansSerif", Font.BOLD, 14));
        g2d.drawString("Chart generation failed", width / 4, height / 2);
        g2d.dispose();
        return toByteArray(image);
    }
}
