package com.arms.api.analysis.scope.service;


import com.arms.api.analysis.scope.model.dto.ScopeDTO;
import com.arms.api.analysis.scope.model.vo.*;
import com.arms.api.product_service.pdservice.model.PdServiceEntity;
import com.arms.api.product_service.pdservice.service.PdService;
import com.arms.api.product_service.pdserviceversion.model.PdServiceVersionEntity;
import com.arms.api.requirement.reqadd.model.entity.ReqAddEntity;
import com.arms.api.requirement.reqadd.model.vo.ReqAddVO;
import com.arms.api.requirement.reqadd.service.ReqAdd;
import com.arms.api.requirement.reqadd_pure.model.ReqAddPureEntity;
import com.arms.api.util.TreeServiceUtils;
import com.arms.api.util.VersionUtil;
import com.arms.api.util.communicate.external.AggregationService;
import com.arms.api.util.communicate.external.response.jira.AlmIssue;
import com.arms.api.util.communicate.internal.InternalService;
import com.arms.api.util.model.dto.PdServiceAndIsReqDTO;
import com.arms.egovframework.javaservice.treeframework.errors.exception.InvalidParamException;
import com.arms.egovframework.javaservice.treeframework.interceptor.SessionUtil;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.hibernate.criterion.Disjunction;
import org.hibernate.criterion.MatchMode;
import org.hibernate.criterion.Restrictions;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;
import org.springframework.util.ObjectUtils;

import java.security.SecureRandom;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static com.arms.api.util.VersionUtil.stringToLongArray;
import static java.util.stream.Collectors.*;

@Service
@RequiredArgsConstructor
@Slf4j
public class ScopeServiceImpl implements ScopeService {
    private static final SecureRandom RANDOM = new SecureRandom();

    private final ReqAdd reqAdd;

    private final PdService pdService;

    private final InternalService internalService;

    private final AggregationService aggregationService;

    @Override
    public NetworkChartVO getReqNetworkFullData(String changeReqTableName, ScopeDTO scopeDTO) throws Exception {
        List<ReqAddPureEntity> reqList = getReqAddPureList(changeReqTableName, scopeDTO);
        List<PdServiceVersionsAndReqVO> pdServiceVersionsAndReq = aggregationService.getPdServiceVersionsAndReq(scopeDTO);
        return generateNetworkChartData(scopeDTO, pdServiceVersionsAndReq,reqList);
    }

    @Override
    public NetworkChartVO getReqNetworkVersionData(String changeReqTableName,ScopeDTO scopeDTO) throws Exception {
        List<ReqAddPureEntity> reqList = getReqAddList(changeReqTableName, scopeDTO);
        List<IssueVO> issueData = aggregationService.getVersionsAndIssueList(scopeDTO);
        return generateNetworkChartWithIssueData(scopeDTO, issueData, reqList);
    }

    @Override
    public NetworkChartVO getReqNetworkIssueData(String changeReqTableName, ScopeDTO scopeDTO) throws Exception {
        List<ReqAddPureEntity> reqList = getReqAddList(changeReqTableName, scopeDTO);
        List<IssueVO> issueData = aggregationService.getIssueList(scopeDTO);
        return generateNetworkChartWithIssueData(scopeDTO,issueData, reqList);
    }

    private List<ReqAddPureEntity> getReqAddPureList(String changeReqTableName, ScopeDTO scopeDTO) throws Exception {
        // 변경 예정.
        SessionUtil.setAttribute("issue-network", changeReqTableName);
        List<ReqAddPureEntity> reqList = getReqAddPureEntities(scopeDTO);
        SessionUtil.removeAttribute("issue-network");
        return reqList;
    }

    private List<ReqAddPureEntity> getReqAddList(String changeReqTableName, ScopeDTO scopeDTO) throws Exception {
        SessionUtil.setAttribute("getReqAddListByFilter", changeReqTableName);
        List<ReqAddPureEntity> reqList = getReqAddPureEntities(scopeDTO);
        SessionUtil.removeAttribute("getReqAddListByFilter");
        return reqList;
    }

    private NetworkChartVO generateNetworkChartData(ScopeDTO scopeDTO, List<PdServiceVersionsAndReqVO> pdServiceVersionsAndReq, List<ReqAddPureEntity> reqList ) throws Exception{

        String pdServiceId = "pdService-" + scopeDTO.getPdServiceAndIsReq().getPdServiceLink();
        List<Long> pdServiceVersionLinks = scopeDTO.getPdServiceAndIsReq().getPdServiceVersionLinks();

        List<NetworkChartVO.Link> links = new ArrayList<>();
        Map<String, NetworkChartVO.Node> nodeMap = new HashMap<>();

        nodeMap.put(pdServiceId, createNode(pdServiceId, String.valueOf(scopeDTO.getPdServiceAndIsReq().getPdServiceLink()),"pdService"));

        pdServiceVersionLinks.forEach(versionLink -> {
            String versionId = "version-" + versionLink;
            links.add(createLink(versionId, pdServiceId)); // 버전과 서비스 링크 생성
            nodeMap.put(versionId,createNode(versionId, String.valueOf(versionLink),"version")); // 버전 노드 생성
        });

        ObjectMapper objectMapper = new ObjectMapper();

        reqList.stream()
                .filter(req -> req.getC_req_pdservice_versionset_link() != null && !req.getC_req_pdservice_versionset_link().trim().isEmpty())
                .forEach(req -> {
                    try {
                        String reqNodeId = "requirement-" + req.getC_id();
                        List<String> versionIdList = objectMapper.readValue(req.getC_req_pdservice_versionset_link(), new TypeReference<List<String>>() {});

                        versionIdList.forEach(versionId -> {
                            NetworkChartVO.Node versionNode = nodeMap.get("version-"+versionId);
                            if (versionNode != null) {
                                links.add(createLink(reqNodeId, versionNode.getId()));
                            }
                        });
                        nodeMap.put(reqNodeId, createNode(reqNodeId, req.getC_title(), "requirement"));
                    } catch (Exception e) {
                        throw new RuntimeException(e.getMessage());
                    }
                });

        pdServiceVersionsAndReq.forEach(reqVO -> {
            Long cReqLink = reqVO.getCReqLink();
            String reqNodeId = "requirement-" + cReqLink;
            NetworkChartVO.Node reqNode = nodeMap.get(reqNodeId);
            if (reqNode != null && reqVO.getReqId() != null) { // reqId (issue의 key 가 not null 조건 추가)
                nodeMap.put(reqVO.getRecentId(), createIssueNode(reqVO));
                links.add(createLink(reqVO.getRecentId(), reqNodeId));
            }
        });

        return NetworkChartVO.builder()
                .links(links)
                .nodes(new ArrayList<>(nodeMap.values()))
                .build();
    }


    private NetworkChartVO generateNetworkChartWithIssueData( ScopeDTO scopeDTO,List<IssueVO> issueData,List<ReqAddPureEntity> reqList) throws Exception {

        // --- 준비 ---
        final String pdServiceId = "pdService-" + scopeDTO.getPdServiceAndIsReq().getPdServiceLink();
        final List<Long> pdServiceVersionLinks = scopeDTO.getPdServiceAndIsReq().getPdServiceVersionLinks();

        List<NetworkChartVO.Link> links = new ArrayList<>();
        Map<String, NetworkChartVO.Node> nodeMap = new HashMap<>();
        Set<String> linkSeen = new HashSet<>();

        // 제품 노드
        nodeMap.put(pdServiceId, createNode(pdServiceId,
                String.valueOf(scopeDTO.getPdServiceAndIsReq().getPdServiceLink()), "pdService"));

        // 버전 노드 및 제품과 sub 연결
        for (Long versionLink : pdServiceVersionLinks) {
            String versionId = "version-" + versionLink;
            nodeMap.putIfAbsent(versionId, createNode(versionId, String.valueOf(versionLink), "version"));
            addLinkDedup(links, linkSeen, versionId, pdServiceId, "sub");
        }

        ObjectMapper om = new ObjectMapper();
        Map<Long, ReqAddPureEntity> reqById = reqList.stream()
                .collect(Collectors.toMap(ReqAddPureEntity::getC_id, Function.identity()));
        Set<Long> aliveReqIds = reqById.keySet();

        for (ReqAddPureEntity req : reqList) {
            Long cId = req.getC_id();
            String reqNodeId = "requirement-" + cId;

            nodeMap.putIfAbsent(reqNodeId, createNode(reqNodeId, req.getC_title(), "requirement"));

            String raw = req.getC_req_pdservice_versionset_link();
            if (raw != null && !raw.trim().isEmpty()) {
                List<String> versions = om.readValue(raw, List.class);
                if (versions != null) {
                    for (String v : versions) {
                        String versionId = "version-" + v;
                        nodeMap.putIfAbsent(versionId, createNode(versionId, v, "version"));
                        addLinkDedup(links, linkSeen, reqNodeId, versionId, "sub");
                    }
                }
            }
        }

        // 이슈 노드 전부 생성
        for (IssueVO issue : issueData) {

            if (Boolean.TRUE.equals(issue.getIsReq())) {
                Long cReqId = issue.getCReqLink();
                if (cReqId == null || !aliveReqIds.contains(cReqId)) {
                    continue; // 삭제된/미매핑 REQ는 스킵
                }
            }
            nodeMap.putIfAbsent(issue.getRecentId(), createIssueNode(issue));
        }

        Map<String, String> issueKeyToRecentId = issueData.stream()
                .filter(i -> i.getIssueKey() != null && !i.getIssueKey().isEmpty())
                .filter(i -> nodeMap.containsKey(i.getRecentId())) // 실제 노드만
                .collect(Collectors.toMap(IssueVO::getIssueKey, IssueVO::getRecentId, (a, b) -> a));

        for (IssueVO issue : issueData) {
            if (Boolean.TRUE.equals(issue.getIsReq()) && issue.getCReqLink() != null) {
                Long cReqId = issue.getCReqLink();
                if (!aliveReqIds.contains(cReqId)) continue;  // 삭제된 요구사항은 연결 금지
                String reqNodeId = "requirement-" + cReqId;
                // 혹시라도 빠졌다면 안전하게 보강
                ReqAddPureEntity req = reqById.get(cReqId);
                if (req != null) {
                    nodeMap.putIfAbsent(reqNodeId, createNode(reqNodeId, req.getC_title(), "requirement"));
                    addLinkDedup(links, linkSeen, issue.getRecentId(), reqNodeId, "sub");
                }
            }
        }

        // SUB 이슈 → 자신의 '직접 부모'(REQ/ALM) 연결 (sub)
        for (IssueVO issue : issueData) {
            if ("SUB".equals(issue.getIssueType())) {
                String parentKey = issue.getParentReqKey();
                if (parentKey != null && !parentKey.isEmpty()) {
                    String parentRecentId = issueKeyToRecentId.get(parentKey);
                    if (parentRecentId != null && nodeMap.containsKey(parentRecentId)) {
                        addLinkDedup(links, linkSeen, issue.getRecentId(), parentRecentId, "sub");
                    }
                }
            }
        }
        for (IssueVO issue : issueData) {
            List<String> linked = issue.getLinkedIssueKeys();
            if (linked == null || linked.isEmpty()) continue;

            for (String other : linked) {
                if (other == null || other.isEmpty()) continue;

                String otherId = nodeMap.containsKey(other) ? other : issueKeyToRecentId.get(other);
                if (otherId == null || otherId.equals(issue.getRecentId())) continue;
                if (!nodeMap.containsKey(otherId)) continue;

                addLinkDedup(links, linkSeen, issue.getRecentId(), otherId, "link");
            }
        }

        return NetworkChartVO.builder()
                .links(links)
                .nodes(new ArrayList<>(nodeMap.values()))
                .build();
    }

    private NetworkChartVO.Link createLink(String source, String target) {
        return NetworkChartVO.Link.builder()
                .source(source)
                .target(target)
                .build();
    }

    private NetworkChartVO.Link createLink(String source, String target, String relation) {
        return NetworkChartVO.Link.builder()
                .source(source)
                .target(target)
                .relation(relation)
                .build();
    }

    private NetworkChartVO.Node createNode(String id, String key ,String type) {
        return NetworkChartVO.Node.builder()
                .id(id)
                .key(key)
                .type(type)
                .build();
    }

    // 요구 사항 노드 생성 메서드
    private NetworkChartVO.Node createIssueNode(PdServiceVersionsAndReqVO reqVO) {

        String key= reqVO.getReqId();

        if (key == null) {
            log.error("key is null, cReqLink => {}", reqVO.getCReqLink());
        }

        if(reqVO.getRelatedIssueCount()>0){
            key += "("+reqVO.getRelatedIssueCount()+")";
        }

        return NetworkChartVO.Node.builder()
                .id(reqVO.getRecentId())
                .key(key)
                .type("REQ")
                .isReq(true)
                .build();
    }
    private NetworkChartVO.Node createIssueNode(IssueVO issueVO) {
        return NetworkChartVO.Node.builder()
                .id(issueVO.getRecentId())
                .type(issueVO.getIssueType())
                .key(issueVO.getIssueKey())
                .parentReqKey(issueVO.getParentReqKey())
                .isReq(issueVO.getIsReq())
                .reqLink(issueVO.getCReqLink())
                .build();
    }

    private void addLinkDedup(List<NetworkChartVO.Link> links,Set<String> seen, String source,String target,String relation) {
        if (source == null || target == null || source.equals(target)) return;
        String a = source.compareTo(target) <= 0 ? source : target;
        String b = source.compareTo(target) <= 0 ? target : source;
        String key = relation + "|" + a + "|" + b;
        if (seen.add(key)) {
            links.add(createLink(source, target, relation));
        }
    }


    @Override
    public List<TreeNodeVO> getReqNetworkFilterData (String changeReqTableName,ScopeDTO scopeDTO) throws Exception{

        SessionUtil.setAttribute("getReqAddListByFilter", changeReqTableName);

        List<ReqAddPureEntity> reqList = getReqAddPureEntities(scopeDTO);

        Map<String, Map<String, List<ReqAddPureEntity>>> serviceVersionMap = new HashMap<>();

        ObjectMapper objectMapper = new ObjectMapper();

        for (ReqAddPureEntity req : reqList) {

            String serviceId = req.getC_req_pdservice_link();
            String versionIds = req.getC_req_pdservice_versionset_link();

            if (versionIds == null || versionIds.trim().isEmpty()) {
                continue;
            }
            List<String> versionIdList = objectMapper.readValue(versionIds, List.class);

            serviceVersionMap.putIfAbsent(serviceId, new HashMap<>());

            Map<String, List<ReqAddPureEntity>> versionMap = serviceVersionMap.computeIfAbsent(serviceId, k -> new HashMap<>());

            for (String versionId : versionIdList) {
                List<ReqAddPureEntity> reqListForVersion = versionMap.computeIfAbsent(versionId, k -> new ArrayList<>());
                reqListForVersion.add(req);
            }
        }

        SessionUtil.removeAttribute("getReqAddListByFilter");

        return serviceVersionMap.entrySet().stream()
                .map(serviceEntry -> new TreeNodeVO(
                        "service_" + serviceEntry.getKey(),
                        List.of(String.valueOf(serviceEntry.getKey())),
                        "service",
                        new TreeNodeVO.TreeNodeAttributes("service", String.valueOf(serviceEntry.getKey()), String.valueOf(serviceEntry.getKey())),
                        "opened",
                        0,
                        serviceEntry.getValue().entrySet().stream()
                                .map(versionEntry -> new TreeNodeVO(
                                        "version_" + versionEntry.getKey(),
                                        List.of(String.valueOf(versionEntry.getKey())),
                                        "version",
                                        new TreeNodeVO.TreeNodeAttributes("version", String.valueOf(versionEntry.getKey()),versionEntry.getKey()),
                                        "opened",
                                        0,
                                        versionEntry.getValue().stream()
                                                .map(req -> new TreeNodeVO(
                                                        "req_" + req.getId(),
                                                        List.of(req.getC_title()),
                                                        "requirement",
                                                        new TreeNodeVO.TreeNodeAttributes("requirement", String.valueOf(req.getC_id()) , req.getC_title()),
                                                        "opened",
                                                        0,
                                                        null
                                                )).collect(Collectors.toList())
                                )).collect(Collectors.toList())
                )).collect(Collectors.toList());
    }


    @Override
    public List<CircularPackingChartVO> getCircularPackingChartData(String changeReqTableName, ScopeDTO scopeDTO) throws Exception {

        List<ReqAddPureEntity> reqList = getReqAddList(changeReqTableName, scopeDTO);

        List<Long> reqIdList = reqList.stream()
                .map(ReqAddPureEntity::getC_id)
                .filter(Objects::nonNull)
                .collect(Collectors.toList());
        scopeDTO.setReqIdList(reqIdList);

        List<CircularPackingChartVO> reqData = aggregationService.getCircularPackingChartData(scopeDTO);

        Stream<CircularPackingChartVO> almStream = reqData.stream()
                .map(req -> CircularPackingChartVO.builder()
                        .almState(req.getAlmState())
                        .linkedIssueCount(req.getLinkedIssueCount())
                        .subTaskCount(req.getSubTaskCount())
                        .workersList(req.getWorkersList())
                        .recentId(req.getRecentId())
                        .issueKey(req.getIssueKey())
                        .cReqLink(req.getCReqLink())
                        .serviceId(req.getServiceId())
                        .versions(req.getVersions())
                        .type(req.getType())
                        .build()
                );

        Stream<CircularPackingChartVO> armsStream = reqList.stream()
                .map(req -> CircularPackingChartVO.builder()
                        .title(req.getC_title())
                        .stateLink(String.valueOf(req.getC_req_state_link()))
                        .cReqLink(req.getC_id())
                        .versions(parseVersionList(req.getC_req_pdservice_versionset_link()))
                        .serviceId(Long.valueOf(req.getC_req_pdservice_link()))
                        .type("ARMS")
                        .build()
                );
        return Stream.concat(almStream, armsStream)
                .collect(Collectors.toList());
    }


    private List<Long> parseVersionList(String versionSetLink) {
        return Optional.of(VersionUtil.stringToLongArray(versionSetLink))
                        .map(Arrays::asList).orElse(Collections.emptyList());

    }


    private String getVersionNames(Map<Long, String> versionNameMap, Long[] versionArr) {
        String key =
            Arrays.stream(versionArr)
                .map(version-> versionNameMap.getOrDefault(version,""))
                .collect(joining(","));

        log.info("[ScopeServiceImple  :: 버전_요구사항_자료] :: 만들어진Key ==> {}", key);
        return key;
    }

    private PdServiceEntity getPdServiceEntity(Long pdServiceId) throws Exception {
        PdServiceEntity pdServiceEntity = new PdServiceEntity();
        pdServiceEntity.setC_id(pdServiceId);
        return pdService.getNode(pdServiceEntity);
    }

    /// reqStatusesByVersionWithAssignees END

    @Override
    public List<CountByVersionNamesVO> countByVersionNames(String changeReqTableName, ScopeDTO scopeDTO) throws Exception {

        Long pdServiceId = scopeDTO.getPdServiceAndIsReq().getPdServiceLink();

        SessionUtil.setAttribute("getReqAddListByFilter", changeReqTableName);

        Map<Long, String> verserionNameMap = getPdServiceEntity(pdServiceId).versionNameMap();

        List<ReqAddEntity> reqAddEntities = getReqAddEntitiesWithDate(scopeDTO);

        Map<String, Long> countByVersionNamesMap = reqAddEntities
            .stream()
            .collect(groupingBy(entity -> getVersionNames(verserionNameMap, stringToLongArray(entity.getC_req_pdservice_versionset_link())), counting()));

        SessionUtil.removeAttribute("getReqAddListByFilter");
        log.info("[ScopeServiceImple  :: 버전_요구사항_자료] :: 버전_요구사항_맵 ==> {}", countByVersionNamesMap);

        return countByVersionNames(countByVersionNamesMap);
    }

    private List<CountByVersionNamesVO> countByVersionNames(Map<String, Long> countByVersionNamesMap) {
            return countByVersionNamesMap.entrySet().stream()
                    .map(CountByVersionNamesVO::new).collect(toList());
    }

    @Override
    public List<ReqAddWithPropertyVO> reqAddWithPropertyListPerVersion(String changeReqTableName, Long pdServiceId, List<Long> pdServiceVersionLinks) throws Exception {
        SessionUtil.setAttribute("req-property-list", changeReqTableName);

        Map<Long, String> versionNameMap = getPdServiceEntity(pdServiceId).versionNameMap();

        List<ReqAddWithPropertyVO> voList = getReqAddEntities(pdServiceVersionLinks).stream()
                .map(reqAddEntity -> {
                    String versionNames = getVersionNames(versionNameMap, stringToLongArray(reqAddEntity.getC_req_pdservice_versionset_link()));
                    return ReqAddWithPropertyVO.builder()
                            .pdServiceId(pdServiceId)
                            .reqId(reqAddEntity.getC_id())
                            .versionNames(versionNames)
                            .reqTitle(reqAddEntity.getC_title())
                            .priorityName(reqAddEntity.getReqPriorityEntity().getC_title())
                            .difficultyName(reqAddEntity.getReqDifficultyEntity().getC_title())
                            .stateName(reqAddEntity.getReqStateEntity().getC_title())
                            .build();
                }).collect(toList());

        SessionUtil.removeAttribute("req-property-list");

        return voList;
    }


    private List<ReqAddEntity> getReqAddEntities(List<Long> pdServiceVersionLinks) throws Exception {

        ReqAddEntity reqAddEntity = new ReqAddEntity();

        Disjunction orCondition = Restrictions.disjunction();
        for (Long version : pdServiceVersionLinks) {
            orCondition.add(Restrictions.like("c_req_pdservice_versionset_link", "\\\"" + String.valueOf(version) + "\\\"", MatchMode.ANYWHERE));
        }
        reqAddEntity.getCriterions().add(orCondition);

        reqAddEntity.getCriterions().add(Restrictions.eq("c_type", "default"));

        return reqAdd.getChildNode(reqAddEntity);
    }

    private List<ReqAddEntity> getReqAddEntitiesWithDate(ScopeDTO scopeDTO) throws Exception {

        List<Long> pdServiceVersionLinks = scopeDTO.getPdServiceAndIsReq().getPdServiceVersionLinks();

        ReqAddEntity reqAddEntity = new ReqAddEntity();

        Disjunction orCondition = Restrictions.disjunction();
        for (Long version : pdServiceVersionLinks) {
            orCondition.add(Restrictions.like("c_req_pdservice_versionset_link", "\\\"" + String.valueOf(version) + "\\\"", MatchMode.ANYWHERE));
        }
        reqAddEntity.getCriterions().add(orCondition);

        reqAddEntity.getCriterions().add(Restrictions.eq("c_type", "default"));

        SimpleDateFormat inputFormat = new SimpleDateFormat("yy-MM-dd");
        SimpleDateFormat outputFormat = new SimpleDateFormat("yyyy-MM-dd");

        Date startDate = null;
        Date endDate = null;

        if (scopeDTO.getStartDate() != null) {
            Date tempDate = inputFormat.parse(scopeDTO.getStartDate());
            startDate = outputFormat.parse(outputFormat.format(tempDate));
        }
        if (scopeDTO.getEndDate() != null) {
            Date tempDate = inputFormat.parse(scopeDTO.getEndDate());
            endDate = outputFormat.parse(outputFormat.format(tempDate));

            Calendar calendar = Calendar.getInstance();
            calendar.setTime(endDate);
            calendar.set(Calendar.HOUR_OF_DAY, 23);
            calendar.set(Calendar.MINUTE, 59);
            calendar.set(Calendar.SECOND, 59);
            endDate = calendar.getTime();
        }

        if (startDate != null && endDate != null) {
            reqAddEntity.getCriterions().add(Restrictions.between("c_req_create_date", startDate, endDate));
        } else if (startDate != null) {
            reqAddEntity.getCriterions().add(Restrictions.ge("c_req_create_date", startDate));
        } else if (endDate != null) {
            reqAddEntity.getCriterions().add(Restrictions.le("c_req_create_date", endDate));
        }

        return reqAdd.getChildNode(reqAddEntity);
    }

    private List<ReqAddPureEntity> getReqAddPureEntities(ScopeDTO scopeDTO) throws Exception {

        List<Long> pdServiceVersionLinks = scopeDTO.getPdServiceAndIsReq().getPdServiceVersionLinks();

        ReqAddPureEntity reqAddEntity = new ReqAddPureEntity();

        Disjunction orCondition = Restrictions.disjunction();

        if(scopeDTO.getReqId() != null && !scopeDTO.getReqId().isEmpty()){
            orCondition.add(Restrictions.eq("c_id", Long.parseLong(scopeDTO.getReqId())));
        }else{
            for (Long version : pdServiceVersionLinks) {
                orCondition.add(Restrictions.like("c_req_pdservice_versionset_link", "\\\"" + String.valueOf(version) + "\\\"", MatchMode.ANYWHERE));
            }
        }

        reqAddEntity.getCriterions().add(orCondition);

        reqAddEntity.getCriterions().add(Restrictions.eq("c_type", "default"));

        return reqAdd.getChildNode(reqAddEntity);
    }

    @Override
    public List<ReqDataTableIssuesVO> getDataTableIssues(ScopeDTO scopeDTO) throws Exception{
        return aggregationService.getDataTableIssues(scopeDTO);
    }


    @Override
    public List<TreeBarVO> treeBarData(ScopeDTO scopeDTO) throws Exception {
        Long pdServiceId = scopeDTO.getPdServiceAndIsReq().getPdServiceLink();

        // 1. 제품 조회
        PdServiceEntity pdServiceEntity = TreeServiceUtils
                .getNodeOptional(pdService, pdServiceId, PdServiceEntity.class).orElseThrow(()-> {
                    throw new InvalidParamException("ScopeServiceImpl :: treebar :: product is null");
                });

        // 2. 제품 등록
        List<TreeBarVO> treeBars =  new ArrayList<>(List.of(createTreeBarVO(pdServiceEntity)));

        // 3. 엔진 통신 및 VO 목록 조회
        List<TreeBarIssueVO> treeBarIssueVOS = Optional.ofNullable(
                aggregationService.treeBarData(scopeDTO).getBody()
        ).orElseGet(() -> {
            log.info("No data returned from treeBarTopData");
            return Collections.emptyList();
        });

        assert treeBarIssueVOS != null;

        // 4. reqLink 목록 추출
        List<Long> cReqLinksTop10 = treeBarIssueVOS
                .stream()
                .map(TreeBarIssueVO::getCReqLink)
                .distinct()  // 중복 제거
                .collect(toList());

        // 4. cReqLinks 값을 이용하여 요구사항(REQADD) 10개 조회
        List<ReqAddVO> reqAddVOS = Optional.ofNullable(
                internalService.reqAddList("T_ARMS_REQADD_" + pdServiceId, cReqLinksTop10).getBody()
        ).orElseGet(() -> {
            log.info("No data returned from reqAddList");
            return Collections.emptyList();
        });

        // 5. 제품 버전 등록
        treeBars.addAll(getTreeBarVOSWithVersionLinks(pdServiceEntity, scopeDTO, reqAddVOS));

        // 6. 요구사항 등록
        treeBars.addAll(getReqAddVOToTreeBarVO(reqAddVOS));

        // 7. 담당자 등록
        treeBars.addAll(getAssignees(treeBarIssueVOS, reqAddVOS));

        return treeBars;
    }


    private List<TreeBarVO> getTreeBarVOSWithVersionLinks(PdServiceEntity pdServiceEntity,ScopeDTO scopeDTO,List<ReqAddVO> reqAddVOS) {
        PdServiceAndIsReqDTO pdServiceAndIsReq = scopeDTO.getPdServiceAndIsReq();
        List<Long> pdServiceVersionLinks = pdServiceAndIsReq.getPdServiceVersionLinks();

        return getReqAddVOToTreeBarVO(reqAddVOS)
            .stream()
            .map(filteredRequirement -> {
                Set<Long> reqVersionLinks = filteredRequirement.reqVersionLinks();
                return TreeBarVO.builder()
                    .id(filteredRequirement.getParent())
                    .name(getReqVersionName(pdServiceEntity.versionList(pdServiceVersionLinks), reqVersionLinks))
                    .color("")
                    .type("version")
                    .parent(String.valueOf(pdServiceEntity.getC_id()))
                    .build();
            }).distinct()
            .collect(toList());
    }

    private TreeBarVO createTreeBarVO(PdServiceEntity product) {
        return TreeBarVO.builder()
            .id(product.getC_id().toString())
            .name(product.getC_title())
            .type("product")
            .color("")
            .parent("")
            .build();
    }

    private List<TreeBarVO> getReqAddVOToTreeBarVO(List<ReqAddVO> reqAddVOS) {
        return Optional.ofNullable(reqAddVOS).orElse(Collections.emptyList())
            .stream()
            .map(TreeBarVO::new)
            .collect(toList());
    }

    private String getReqVersionName(List<PdServiceVersionEntity> productVersions, Set<Long> reqVersionLinks) {
        return productVersions
                .stream()
                .filter(version -> reqVersionLinks.contains(version.getC_id()))
                .map(PdServiceVersionEntity::getC_title)
                .collect(joining(","));
    }

    private List<TreeBarVO> getAssignees(List<TreeBarIssueVO> treeBarIssueVOS, List<ReqAddVO> reqAddVOS) {
        Map<String, String> assigneeToColorMap = new HashMap<>();

        Set<Long> reqAddCidSet
                = Optional.ofNullable(reqAddVOS).orElse(Collections.emptyList())
                .stream().map(ReqAddVO::getC_id).collect(toSet());

        return treeBarIssueVOS.stream()
                .filter(issueVO -> reqAddCidSet.contains(issueVO.getCReqLink()))
                .map(top10Requirement -> {
                    String parent = "requirement-" + top10Requirement.getCReqLink();

                    String name = top10Requirement.getAssigneeDisplayName();
                    String color = assigneeToColorMap.computeIfAbsent(name, k ->
                        String.format("#%02x%02x%02x", RANDOM.nextInt(256), RANDOM.nextInt(256), RANDOM.nextInt(256))
                    );
                    long value = top10Requirement.getAssigneeCount();
                    return TreeBarVO.builder()
                        .id(parent + top10Requirement.getAssigneeDisplayName())
                        .parent(parent)
                        .type("assignee")
                        .name(name + " (" + value + ")")
                        .value(value)
                        .color(color)
                        .build();
                })
                .collect(toList());
    }

    @Override
    public List<NetworkChartExcelDataVO> getNetworkChartExcelData(ScopeDTO scopeDTO) throws Exception{
        return aggregationService.getNetworkChartExcelData(scopeDTO);
    }

}
