package com.arms.api.requirement.reqstatus_calendar.service;

import com.arms.api.jira.jiraproject_pure.model.JiraProjectPureEntity;
import com.arms.api.jira.jiraproject_pure.service.JiraProjectPure;
import com.arms.api.product_service.pdservice.model.PdServiceAndVersionListDTO;
import com.arms.api.product_service.pdserviceversion.model.PdServiceVersionEntity;
import com.arms.api.requirement.reqstatus_calendar.model.CalendarDataDTO;
import com.arms.api.requirement.reqstatus_calendar.model.ReqStatusCalendarRequestDTO;
import com.arms.api.requirement.reqstatus_calendar.model.ReqStatusCalendarResponseDTO;
import com.arms.api.util.communicate.external.EngineService;
import com.arms.api.util.communicate.external.response.jira.AlmIssue;
import com.arms.api.util.communicate.internal.InternalService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;

import java.time.LocalDateTime;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

@Slf4j
@Service
@RequiredArgsConstructor
public class ReqStatusCalendarServiceImpl implements ReqStatusCalendarService {

    private final EngineService engineService;
    private final InternalService internalService;
    private final JiraProjectPure jiraProjectPure;

    private static final List<DateTimeFormatter> DATE_TIME_FORMATTERS = Arrays.asList(
            DateTimeFormatter.ofPattern("yyyy-MM-dd'T'HH:mm:ss.SSSZ"),
            DateTimeFormatter.ofPattern("yyyy-MM-dd'T'HH:mm:ss.SSSXXX"),
            DateTimeFormatter.ofPattern("yyyy-MM-dd'T'HH:mm:ssZ"),
            DateTimeFormatter.ofPattern("yyyy-MM-dd'T'HH:mm:ssXXX")
    );

    private static final DateTimeFormatter OUTPUT_DATE_FORMATTER = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");

    @Override
    public List<CalendarDataDTO> getCalendarData(ReqStatusCalendarRequestDTO request) throws Exception {

        Function<ReqStatusCalendarRequestDTO,
                ResponseEntity<ReqStatusCalendarResponseDTO>> fetchAndDeduplicate = dto -> {
            ResponseEntity<ReqStatusCalendarResponseDTO> responseEntity = engineService.이슈목록_가져오기_calendar(dto);
            return responseEntity;
        };

        return processCalendarDataMapping(request, fetchAndDeduplicate);
    }

    private List<CalendarDataDTO> processCalendarDataMapping(
            ReqStatusCalendarRequestDTO request,
            Function<ReqStatusCalendarRequestDTO,
                    ResponseEntity<ReqStatusCalendarResponseDTO>> function
    ) throws Exception {

        ResponseEntity<List<PdServiceAndVersionListDTO>> internalResponse = internalService.getPdServiceEntityAndVersionList();

        if (internalResponse.getBody() == null) {
            log.error("Product service and version information is null.");
            return Collections.emptyList();
        }

        Map<Long, String> pdServiceMap = new HashMap<>();
        Map<Long, String> versionMap = new HashMap<>();
        extractProductAndVersionInfo(internalResponse.getBody(), pdServiceMap, versionMap);

        prepareAlmProjectUrls(request);

        ResponseEntity<ReqStatusCalendarResponseDTO> calendarDataFromEngine = function.apply(request);
        if (calendarDataFromEngine.getBody() == null || calendarDataFromEngine.getBody().getIssueEntityList().isEmpty()) {
            log.info("Issue list from engine is empty.");
            return Collections.emptyList();
        }

        List<AlmIssue> allIssues = calendarDataFromEngine.getBody().getIssueEntityList();
        Map<String, List<AlmIssue>> issuesByParent = allIssues.stream()
                .filter(issue -> issue.getParentReqKey() != null)
                .collect(Collectors.groupingBy(AlmIssue::getParentReqKey));

        List<AlmIssue> requirementIssues = filterAndSortRequirementIssues(allIssues);

        List<CalendarDataDTO> result = new ArrayList<>();
        for (AlmIssue reqIssue : requirementIssues) {
            result.add(mapToCalendarData(pdServiceMap, versionMap, reqIssue));

            List<AlmIssue> subAndLinkedIssues = issuesByParent.getOrDefault(reqIssue.getKey(), Collections.emptyList());
            Set<String> linkedIssueIds = Optional.ofNullable(reqIssue.getLinkedIssues())
                    .map(Arrays::stream)
                    .map(stream -> stream.collect(Collectors.toSet()))
                    .orElse(Collections.emptySet());

            for (AlmIssue subOrLinkedIssue : subAndLinkedIssues) {
                if (linkedIssueIds.contains(subOrLinkedIssue.getRecentId())) {
                    result.add(mapToLinkedIssueCalendarData(pdServiceMap, versionMap, subOrLinkedIssue, reqIssue));
                } else {
                    result.add(mapToSubIssueCalendarData(pdServiceMap, versionMap, subOrLinkedIssue, reqIssue));
                }
            }
        }

        // 최종 결과에서 중복 제거
        return result.stream()
                .collect(Collectors.toMap(
                        dto -> dto.getKey() + "::" + dto.getUpdatedDate(),  // 중복 체크를 위한 키
                        Function.identity(),
                        (existing, newer) -> newer  // 중복시 최신 데이터 유지
                ))
                .values()
                .stream()
                .collect(Collectors.toList());
    }

    private void extractProductAndVersionInfo(List<PdServiceAndVersionListDTO> serviceAndVersionList,
                                              Map<Long, String> pdServiceMap, Map<Long, String> versionMap) {
        for (PdServiceAndVersionListDTO dto : serviceAndVersionList) {
            pdServiceMap.put(dto.getC_id(), dto.getC_title());
            versionMap.putAll(buildVersionIdNameMap(dto.getPdServiceVersionEntityList()));
        }
    }

    private void prepareAlmProjectUrls(ReqStatusCalendarRequestDTO request) throws Exception {
        List<Long> almProjectIds = Optional.ofNullable(request.getAlmProjectIds()).orElse(Collections.emptyList());
        if (!almProjectIds.isEmpty()) {
            List<JiraProjectPureEntity> almProjects = jiraProjectPure.getJiraProjects(almProjectIds);
            List<String> almProjectUrls = almProjects.stream()
                    .map(JiraProjectPureEntity::getC_jira_url)
                    .collect(Collectors.toList());
            request.setAlmProjectUrls(almProjectUrls);
            log.info("ALM Project URLs set: {}", almProjectUrls);
        }
    }

    private List<AlmIssue> filterAndSortRequirementIssues(List<AlmIssue> issues) {
        return issues.stream()
                .filter(issue -> Boolean.TRUE.equals(issue.getIsReq()))
                .peek(this::formatIssueDates)
                .sorted(Comparator.comparing(AlmIssue::getUpdated, Comparator.nullsLast(Comparator.reverseOrder())))
                .collect(Collectors.toList());
    }

    private void formatIssueDates(AlmIssue issue) {
        issue.setCreated(formatDate(issue.getCreated()));
        issue.setUpdated(formatDate(issue.getUpdated()));
        issue.setResolutiondate(formatDate(issue.getResolutiondate()));
    }

    private String formatDate(String inputDate) {
        if (inputDate == null || inputDate.isEmpty()) {
            return "";
        }

        for (DateTimeFormatter formatter : DATE_TIME_FORMATTERS) {
            try {
                return ZonedDateTime.parse(inputDate, formatter).format(OUTPUT_DATE_FORMATTER);
            } catch (DateTimeParseException e) {
                // Try next formatter
                // log.error(e.getMessage(), e);
            }
        }

        try {
            LocalDateTime.parse(inputDate, OUTPUT_DATE_FORMATTER);
            return inputDate;
        } catch (DateTimeParseException e) {
            // Not in the output format either
            // log.error(e.getMessage(), e);
        }

        log.error("Unsupported date format: {}", inputDate);
        return inputDate;
    }

    private CalendarDataDTO mapToCalendarData(Map<Long, String> pdServiceMap, Map<Long, String> versionMap, AlmIssue issue) {
        CalendarDataDTO baseCalendarData = createBaseCalendarDataBuild(pdServiceMap, versionMap, issue);

        if (Boolean.TRUE.equals(issue.getIsReq())) {
            baseCalendarData.setIsReq(Boolean.TRUE);
            baseCalendarData.setReqTitle(issue.getSummary());
            baseCalendarData.setReqState(Optional.ofNullable(issue.getCReqProperty())
                    .map(AlmIssue.암스_요구사항_속성::getCReqStateName)
                    .orElse(""));
            baseCalendarData.setCreateDate(Optional.ofNullable(issue.getCreated()).orElse("생성일 정보 없음"));
            baseCalendarData.setUpdatedDate(Optional.ofNullable(issue.getUpdated()).orElse("수정일 정보 없음"));
            baseCalendarData.setResolutionDate(Optional.ofNullable(issue.getResolutiondate()).orElse(""));
        }

        return baseCalendarData;
    }

    private CalendarDataDTO mapToLinkedIssueCalendarData(Map<Long, String> pdServiceMap, Map<Long, String> versionMap, AlmIssue linkedIssue, AlmIssue parentReq) {
        formatIssueDates(linkedIssue);
        AlmIssue issueAsLinked = linkedIssue.create연결이슈();
        String etcContent = parentReq.getKey() + "의 연결이슈";
        issueAsLinked.setEtc(etcContent);

        CalendarDataDTO baseCalendarData = createBaseCalendarDataBuild(pdServiceMap, versionMap, issueAsLinked);
        baseCalendarData.setIsReq(Boolean.FALSE);
        baseCalendarData.setReqTitle(parentReq.getSummary());
        baseCalendarData.setReqState(Optional.ofNullable(parentReq.getCReqProperty())
                .map(AlmIssue.암스_요구사항_속성::getCReqStateName)
                .orElse(""));
        baseCalendarData.setCreateDate(Optional.ofNullable(issueAsLinked.getCreated()).orElse("생성일 정보 없음"));
        baseCalendarData.setUpdatedDate(Optional.ofNullable(issueAsLinked.getUpdated()).orElse("수정일 정보 없음"));
        baseCalendarData.setResolutionDate(Optional.ofNullable(issueAsLinked.getResolutiondate()).orElse(""));
        baseCalendarData.setEtc(etcContent);

        return baseCalendarData;// ✅ DTO 완성해서 리턴
    }

    private CalendarDataDTO mapToSubIssueCalendarData(Map<Long, String> pdServiceMap, Map<Long, String> versionMap, AlmIssue subIssue, AlmIssue parentReq) {
        formatIssueDates(subIssue);

        String isReqName = determineSubIssueName(subIssue);

        CalendarDataDTO baseCalendarData = createBaseCalendarDataBuild(pdServiceMap, versionMap, subIssue);

        baseCalendarData.setIsReq(Boolean.FALSE);
        baseCalendarData.setReqTitle(parentReq.getSummary());
        baseCalendarData.setReqState(Optional.ofNullable(parentReq.getCReqProperty())
                .map(AlmIssue.암스_요구사항_속성::getCReqStateName)
                .orElse(""));
        baseCalendarData.setCreateDate(Optional.ofNullable(subIssue.getCreated()).orElse("생성일 정보 없음"));
        baseCalendarData.setUpdatedDate(Optional.ofNullable(subIssue.getUpdated()).orElse("수정일 정보 없음"));
        baseCalendarData.setResolutionDate(Optional.ofNullable(subIssue.getResolutiondate()).orElse(""));

        return baseCalendarData; // ✅ DTO 완성해서 리턴
    }

    private CalendarDataDTO createBaseCalendarDataBuild(Map<Long, String> pdServiceMap, Map<Long, String> versionMap, AlmIssue issue) {
        String versionNames = Optional.ofNullable(issue.getPdServiceVersions())
                .map(versions -> Arrays.stream(versions)
                        .map(versionMap::get)
                        .filter(Objects::nonNull)
                        .collect(Collectors.joining(",")))
                .filter(s -> !s.isEmpty())
                .orElse("버전 정보 없음");

        return CalendarDataDTO.builder()
                .pdServiceId(issue.getPdServiceId())
                .pdServiceName(Optional.ofNullable(pdServiceMap.get(issue.getPdServiceId())).orElse("제품(서비스) 정보 없음"))
                .pdServiceVersionNames(versionNames)
                .cReqLink(issue.getCReqLink())
                .upperKey(issue.getUpperKey())
                .parentReqKey(issue.getParentReqKey())
                .almProjectName(issue.projectName())
                .key(issue.getKey())
                .issueID(issue.getIssueID())
                .docId(issue.getRecentId())
                .issueTitle(issue.getSummary())
                .issueStatus(issue.statusName())
                .assigneeName(Optional.ofNullable(issue.getAssignee()).map(AlmIssue.담당자::getDisplayName).orElse("담당자 정보 없음"))
                .assigneeEmail(Optional.ofNullable(issue.getAssignee()).map(AlmIssue.담당자::getEmailAddress).orElse("담당자 메일 없음"))
                .build();  // ✅ DTO 완성해서 리턴
    }

    private String determineSubIssueName(AlmIssue issue) {
        if (!Objects.equals(issue.getParentReqKey(), issue.getUpperKey()) && issue.getUpperKey() != null) {
            return issue.getUpperKey() + "의 하위이슈";
        }
        return issue.getParentReqKey() + "의 하위이슈";
    }

    private Map<Long, String> buildVersionIdNameMap(List<PdServiceVersionEntity> versionEntities) {
        DateTimeFormatter inputFormatter = DateTimeFormatter.ofPattern("yyyy/MM/dd HH:mm");
        DateTimeFormatter outputFormatter = DateTimeFormatter.ofPattern("yyyy-MM-dd");

        return versionEntities.stream()
                .collect(Collectors.toMap(
                        PdServiceVersionEntity::getC_id,
                        entity -> {
                            LocalDateTime startDateTime = LocalDateTime.parse(entity.getC_pds_version_start_date(), inputFormatter);
                            LocalDateTime endDateTime = LocalDateTime.parse(entity.getC_pds_version_end_date(), inputFormatter);

                            String startDate = startDateTime.format(outputFormatter);
                            String endDate = endDateTime.format(outputFormatter);
                            return entity.getC_title() + " (" + startDate + " ~ " + endDate + ")";
                        },
                        (existingValue, newValue) -> newValue
                ));
    }
}
