package com.arms.api.requirement.reqadd.kafka;

import com.arms.api.requirement.reqadd.model.dto.ReqAddDTO;
import com.arms.api.util.DataSerializer;
import com.arms.api.util.communicate.internal.InternalService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.springframework.kafka.annotation.KafkaListener;
import org.springframework.kafka.support.Acknowledgment;
import org.springframework.stereotype.Component;

import java.time.Duration;
import java.time.Instant;

/**
 * 요구사항(Requirement) 추가 Kafka Consumer
 * 메시지 처리 추적 기능 포함
 */
@Slf4j
@Component
@RequiredArgsConstructor
public class ReqAddConsumer {

    private final InternalService internalService;

    @javax.annotation.PostConstruct
    public void init() {
        log.info("========================================");
        log.info("✅ ReqAddConsumer Bean initialized successfully!");
        log.info("📡 Listening on topic: ${spring.kafka.topic.reqadd:REQADD}");
        log.info("👥 Consumer group: requirement-consumer-group");
        log.info("========================================");
    }

    @KafkaListener(
            topics = "${spring.kafka.topic.reqadd:REQADD}",
            containerFactory = "reqAddKafkaListenerContainerFactory"
    )
    public void consumeReqAddAddNodeRequest(
            ConsumerRecord<String, String> record,
            Acknowledgment ack
    ) {
        Instant startTime = Instant.now();
        String key = record.key();
        String uuid = "unknown";

        try {
            // ============================================
            // 1. 메시지 수신 로그
            // ============================================
            log.info("━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━");
            log.info("🟡 [RECEIVED] Kafka message received");
            log.info("   Key: {}", key);
            log.info("   Partition: {}, Offset: {}", record.partition(), record.offset());
            log.info("   Timestamp: {}", Instant.ofEpochMilli(record.timestamp()));

            // ============================================
            // 2. Key 파싱 및 검증
            // ============================================
            if (key == null || key.isEmpty()) {
                log.error("❌ [INVALID] Received message with null or empty key");
                ack.acknowledge();
                return;
            }

            String[] parts = key.split(":");

            if (parts.length < 4) {
                log.error("❌ [INVALID] Invalid key format: {}. Expected: requirement:tableName:eventType:uuid", key);
                ack.acknowledge();
                return;
            }

            String messageType = parts[0];  // "requirement"
            String tableName = parts[1];     // "T_ARMS_REQADD"
            String eventType = parts[2];     // "CREATE" | "UPDATE" | "DELETE"
            uuid = parts[3];                 // UUID

            log.info("📋 [PARSED] Message details:");
            log.info("   Type: {}", messageType);
            log.info("   Table: {}", tableName);
            log.info("   Event: {}", eventType);
            log.info("   UUID: {}", uuid);

            // ============================================
            // 3. Payload 역직렬화
            // ============================================
            String value = record.value();
            log.info("📦 [PAYLOAD] Raw JSON: {}", value);

            ReqAddDTO payload = DataSerializer.deserialize(value, ReqAddDTO.class);

            if (payload == null) {
                log.error("❌ [FAILED] Failed to deserialize payload - UUID: {}", uuid);
                ack.acknowledge();
                return;
            }

            log.info("✅ [DESERIALIZED] Payload: {}", payload);

            // ============================================
            // 4. 이벤트 타입별 처리
            // ============================================
            log.info("🔄 [PROCESSING] Starting {} processing - UUID: {}", eventType, uuid);

            switch (eventType) {
                case "CREATE":
                    // TODO: 실제 로직 활성화
                    // internalService.create(tableName, payload);

                    log.info("   [CREATE] Processing create event");
                    log.info("   Table: {}", tableName);
                    log.info("   Data: {}", payload);
                    break;

                case "UPDATE":
                    // TODO: 실제 로직 활성화
                    // internalService.update(tableName, payload);

                    log.info("   [UPDATE] Processing update event");
                    log.info("   Table: {}", tableName);
                    log.info("   Data: {}", payload);
                    break;

                case "DELETE":
                    // TODO: 실제 로직 활성화
                    // internalService.delete(tableName, payload);

                    log.info("   [DELETE] Processing delete event");
                    log.info("   Table: {}", tableName);
                    log.info("   Data: {}", payload);
                    break;

                default:
                    log.warn("⚠️  [UNKNOWN] Unknown event type: {} - UUID: {}", eventType, uuid);
                    ack.acknowledge();
                    return;
            }

            // ============================================
            // 5. 처리 성공
            // ============================================
            Duration processingTime = Duration.between(startTime, Instant.now());

            log.info("✅ [SUCCESS] Message processed successfully");
            log.info("   UUID: {}", uuid);
            log.info("   Processing time: {} ms", processingTime.toMillis());
            log.info("   Event: {}", eventType);

        } catch (Exception e) {
            // ============================================
            // 6. 처리 실패
            // ============================================
            Duration processingTime = Duration.between(startTime, Instant.now());

            log.error("❌ [FAILED] Message processing failed");
            log.error("   UUID: {}", uuid);
            log.error("   Key: {}", key);
            log.error("   Processing time: {} ms", processingTime.toMillis());
            log.error("   Error: {}", e.getMessage(), e);

            // TODO: Dead Letter Queue 전송 또는 재시도 로직
        }

        // ============================================
        // 7. Offset 커밋
        // ============================================
        ack.acknowledge(); // → 메시지 처리 완료 후 **수동으로 커밋**하여 처리 순서를 보장합니다.
        log.info("🟢 [COMMITTED] Offset committed - UUID: {}", uuid);
        log.info("━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━");
    }
}