package com.arms.api.report.export_service.controller;

import com.arms.api.report.export_service.dto.ReportExportRequest;
import com.arms.api.report.export_service.dto.ReportExportResponse;
import com.arms.api.report.export_service.enums.ExportType;
import com.arms.api.report.export_service.model.ReportData;
import com.arms.api.report.export_service.service.ReportExportService;
import com.arms.api.report.export_service.service.ReportExportServiceFactory;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;

/**
 * PPT 리포트 내보내기 컨트롤러
 *
 * @author HS.Yang
 * @since 25.12.03
 */
@Slf4j
@RestController
@RequestMapping("/arms/export/reports")
@RequiredArgsConstructor
@Tag(name = "PPT Export", description = "PPT 내보내기 API")
public class ReportExportController {

    private final ReportExportServiceFactory exportServiceFactory;

    /**
     * PPT로 내보내기
     */
    @Operation(summary = "PPT로 내보내기")
    @PostMapping("/export/ppt")
    public ResponseEntity<byte[]> exportToPpt(@Valid @RequestBody ReportExportRequest request) {
        log.info("PPT Export request received - title: {}", request.getTitle());

        ReportExportService exportService = exportServiceFactory.getService(ExportType.PPT);
        ReportData reportData = request.toReportData();
        byte[] content = exportService.export(reportData);
        String filename = generateFilename(reportData.getTitle(), exportService.getFileExtension());

        log.info("PPT exported - filename: {}, size: {} bytes", filename, content.length);

        return ResponseEntity.ok()
                .header(HttpHeaders.CONTENT_DISPOSITION, "attachment; filename=\"" + filename + "\"")
                .header(HttpHeaders.ACCESS_CONTROL_EXPOSE_HEADERS, HttpHeaders.CONTENT_DISPOSITION)
                .contentType(MediaType.parseMediaType(exportService.getContentType()))
                .contentLength(content.length)
                .body(content);
    }

    /**
     * PPT 메타데이터 조회
     */
    @Operation(summary = "PPT 메타데이터 조회")
    @PostMapping("/export/ppt/metadata")
    public ResponseEntity<ReportExportResponse> exportPptMetadata(
            @Valid @RequestBody ReportExportRequest request) {

        ReportExportService exportService = exportServiceFactory.getService(ExportType.PPT);
        ReportData reportData = request.toReportData();
        byte[] content = exportService.export(reportData);
        String filename = generateFilename(reportData.getTitle(), exportService.getFileExtension());

        return ResponseEntity.ok(ReportExportResponse.success(
                reportData.getReportId(), filename, ExportType.PPT, content.length));
    }

    /**
     * 파일명 생성
     */
    private String generateFilename(String title, String extension) {
        String safeName = title != null ?
                title.replaceAll("[^a-zA-Z0-9가-힣\\s]", "").replaceAll("\\s+", "_") :
                "report";
        String timestamp = LocalDateTime.now().format(DateTimeFormatter.ofPattern("yyyyMMdd_HHmmss"));
        String filename = safeName + "_" + timestamp + extension;
        return URLEncoder.encode(filename, StandardCharsets.UTF_8).replace("+", "%20");
    }
}
